package com.tydic.dyc.ubc.repository.impl;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.base.utils.JUtil;
import com.tydic.dyc.ubc.components.UbcProperties;
import com.tydic.dyc.ubc.components.es.UbcElasticsearchUtil;
import com.tydic.dyc.ubc.components.runnable.SyncUserBehaviorToEsRunnable;
import com.tydic.dyc.ubc.components.thread.UbcThreadPool;
import com.tydic.dyc.ubc.constants.UbcRspConstants;
import com.tydic.dyc.ubc.model.common.UbcSnapInfoDo;
import com.tydic.dyc.ubc.model.common.UbcUserBehaviorInfoDo;
import com.tydic.dyc.ubc.model.common.UbcUserOperateInfoDo;
import com.tydic.dyc.ubc.repository.UbcCommonRepository;
import com.tydic.dyc.ubc.repository.dao.*;
import com.tydic.dyc.ubc.repository.po.*;
import com.tydic.dyc.ubc.service.common.bo.UbcBusiDefineBo;
import com.tydic.dyc.ubc.service.common.bo.UbcLocationDefineBo;
import com.tydic.dyc.ubc.service.common.bo.UbcSyncEsServiceReqBo;
import com.tydic.dyc.ubc.service.common.bo.UbcSyncEsServiceRspBo;
import com.tydic.dyc.ubc.utils.*;
import org.joda.time.LocalDate;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.PropertyPlaceholderHelper;

import java.sql.SQLException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 标题:UbcCommonRepositoryImpl
 * 说明:
 * 时间:2023/8/21 11:15
 * 作者:田桂银
 */
@Service
public class UbcCommonRepositoryImpl implements UbcCommonRepository {

    @Autowired
    private UbcRelSnapTempTypeMapper ubcRelSnapTempTypeMapper;
    @Autowired
    private UbcSnapInfoMapper ubcSnapInfoMapper;
    @Autowired
    private UbcBusiDefineMapper ubcBusiDefineMapper;
    @Autowired
    private UbcUserBehaviorInfoMapper ubcUserBehaviorInfoMapper;

    @Autowired
    private UbcUserOperateInfoMapper ubcUserOperateInfoMapper;

    @Autowired
    private UbcLocationDefineMapper ubcLocationDefineMapper;
    @Autowired
    private UbcProperties ubcProperties;

    @Autowired
    private UbcBehaviorDefineMapper ubcBehaviorDefineMapper;

    @Autowired
    private UbcBehaviorOperateDefineMapper ubcBehaviorOperateDefineMapper;

    @Autowired
    private UbcBehaviorOperatePropertyMapper ubcBehaviorOperatePropertyMapper;
    @Autowired
    private UbcBehaviorOperateBasePropertyMapper ubcBehaviorOperateBasePropertyMapper;

    @Autowired
    private UbcThreadPool ubcThreadPool;


    //保存用户操作时间，是否使用后端时间开关（前端时间有误差）
    @Value("${ubc.saveSnapInfo.useServerTime:true}")
    private boolean useServerTime;

    @Value("${spring.datasource.url}")
    private String dataSourceUrl;

    @Value("${spring.datasource.username}")
    private String dataSourceUserName;

    @Value("${spring.datasource.password}")
    private String dataSourcePassword;

    @Value("${spring.datasource.driver-class-name:com.mysql.jdbc.Driver}")
    private String driverClassName;


    @Override
    public UbcSnapInfoDo saveSnapInfo(UbcSnapInfoDo reqDo) {
        UbcSnapInfoDo rspBO = new UbcSnapInfoDo();
        try {
            //插入数据
            rspBO = getUbcSnapInfoDo(reqDo);
        } catch (Exception e) {
            //如果表不存在的情况下，先创建表，然后再插入数据
            if (e.getMessage().contains("Unknown error 1146") || e.getMessage().contains("Unknown database")) {
                try {
                    String snapTempCode = reqDo.getSnapTempCode();
                    //校验快照功能编码，是否存在
                    validateSnapTempCode(snapTempCode);
                    createDatabase(dataSourceUrl, dataSourceUserName, dataSourcePassword, snapTempCode, LocalDate.now().getYear(), ubcProperties.getSnapInfoPrefix());
                    //创建表成功后，插入数据
                    rspBO = getUbcSnapInfoDo(reqDo);
                } catch (Exception ex) {
                    //并发下，防止数据丢失
                    if (ex.getMessage().contains("Unknown error 1050") || ex.getMessage().contains("already exists")) {
                        rspBO = getUbcSnapInfoDo(reqDo);
                        return rspBO;
                    } else {
                        throw new BaseBusinessException(UbcRspConstants.RSP_CODE_SYSTEM_EXCEPTION, ex.getMessage());
                    }
                }
            } else {
                throw new BaseBusinessException(UbcRspConstants.RSP_CODE_SYSTEM_EXCEPTION, e.getMessage());
            }
        }
        return rspBO;
    }

    public UbcSnapInfoDo getUbcSnapInfoDo(UbcSnapInfoDo reqDo) {
        reqDo.setSnapId(IdUtil.nextId());
        String snapTempCode = reqDo.getSnapTempCode();
        //校验快照功能编码，是否存在
        validateSnapTempCode(snapTempCode);
        UbcSnapInfo snapInfo = JUtil.js(reqDo, UbcSnapInfo.class);
        //根据快照表前缀，快照编码，快照id，或者保存的表明名
        snapInfo.setReceiveTime(new Date());
        snapInfo.setTableName(TableUtil.getSnapInfoTableName(ubcProperties.getSnapInfoPrefix(), snapTempCode, reqDo.getSnapId()));
        ubcSnapInfoMapper.insertByTableName(snapInfo);
        return reqDo;
    }

    /**
     * 描述:创建表
     * jdbcUrl：数据库连接地址
     * username：用户名
     * password：密码
     * tableName：后缀名（没有就不传）
     * year：年份
     * prefix：需要新增的表的前缀
     * @param jdbcUrl, username, password, tableName, year, prefix
     * @return void
     */
    private void createDatabase(String jdbcUrl, String username, String password, String tableName, int year, String prefix) {
        StringBuilder createRelPoolBuilder = new StringBuilder();
        if (prefix.equals(ubcProperties.getSnapInfoPrefix())) {
            createRelPoolBuilder.append("CREATE TABLE ubc_snap_info").append("_").append(tableName).append("_").append(year);
            createRelPoolBuilder.append(" LIKE ubc_snap_info").append("_").append(tableName).append("_").append(year - 1);
        } else if (prefix.equals(ubcProperties.getUserBehaviorInfoPrefix())) {
            createRelPoolBuilder.append("CREATE TABLE ubc_user_behavior_info").append("_").append(year);
            createRelPoolBuilder.append(" LIKE ubc_user_behavior_info").append("_").append(year - 1);
        } else if (prefix.equals(ubcProperties.getUserOperateInfoPrefix())) {
            createRelPoolBuilder.append("CREATE TABLE ubc_user_operate_info").append("_").append(year);
            createRelPoolBuilder.append(" LIKE ubc_user_operate_info").append("_").append(year - 1);
        }
        JdbcUtils.executeSql(jdbcUrl, username, password, driverClassName, String.valueOf(createRelPoolBuilder));
    }

    @Override
    public UbcSnapInfoDo selectSnapById(UbcSnapInfoDo reqDo) {
        //校验快照功能编码，是否存在
        String snapTempCode = reqDo.getSnapTempCode();
        validateSnapTempCode(snapTempCode);
        UbcSnapInfo snapInfo = JUtil.js(reqDo, UbcSnapInfo.class);
        //根据快照表前缀，快照编码，快照id，或者保存的表明名
        snapInfo.setTableName(TableUtil.getSnapInfoTableName(ubcProperties.getSnapInfoPrefix(), snapTempCode, reqDo.getSnapId()));
        UbcSnapInfo ubcSnapInfo = ubcSnapInfoMapper.selectSnapById(snapInfo);
        return JUtil.js(ubcSnapInfo, UbcSnapInfoDo.class);
    }

    //此处独立开启事务，避免外部线程池执行时，事务未提交，查询不到数据
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    @Override
    public UbcUserBehaviorInfoDo saveBehaviorInfo(UbcUserBehaviorInfoDo reqDo) {
        Date currentDate = new Date();
        Long behaviorId = reqDo.getBehaviorId();
        if (ObjectUtil.isEmpty(behaviorId)) {
            //查询业务行为定义是否存在，不存在返回报错
            UbcBehaviorDefine ubcBehaviorDefine = ubcBehaviorDefineMapper.selectOne(new LambdaQueryWrapper<UbcBehaviorDefine>().eq(UbcBehaviorDefine::getBehaviorCode, reqDo.getBehaviorCode()));
            if (ObjectUtil.isEmpty(ubcBehaviorDefine)) {
                throw new BaseBusinessException(UbcRspConstants.RSP_CODE_BEHAVIOR_NOT_DEFINE, "用户行为编码[" + reqDo.getBehaviorCode() + "]未定义");
            }

            //校验位置编码是否定义了，同时补充位置名称，没定义直接报错
            if (ObjectUtil.isNotEmpty(reqDo.getLocationCode())) {
                UbcLocationDefine ubcLocationDefine = ubcLocationDefineMapper.selectOne(new LambdaQueryWrapper<UbcLocationDefine>().eq(UbcLocationDefine::getLocationCode, reqDo.getLocationCode()));
                if (ObjectUtil.isEmpty(ubcLocationDefine)) {
                    throw new BaseBusinessException(UbcRspConstants.RSP_CODE_BEHAVIOR_NOT_DEFINE, "位置编码[" + reqDo.getLocationCode() + "]未定义");
                }
                reqDo.setLocationName(ubcLocationDefine.getLocationName());
            }
            //如果行为id为空，则直接创建行为
            behaviorId = IdUtil.nextId();
            reqDo.setBehaviorId(behaviorId);

            //添加用户行为数据，如果不存在表，先创建表，然后再插入数据
            try {
                insertUserBehaviorInfo(reqDo, currentDate, behaviorId);
            } catch (Exception e){
                //如果表不存在的情况下，先创建表，然后再插入数据
                if (e.getMessage().contains("Unknown error 1146") || e.getMessage().contains("Unknown database")) {
                    try {
                        createDatabase(dataSourceUrl, dataSourceUserName, dataSourcePassword, null, LocalDate.now().getYear(), ubcProperties.getUserBehaviorInfoPrefix());
                        //创建表成功后，插入数据
                        insertUserBehaviorInfo(reqDo, currentDate, behaviorId);
                    } catch (Exception ex) {
                        //并发下，防止数据丢失
                        if (ex.getMessage().contains("Unknown error 1050") || ex.getMessage().contains("already exists")) {
                            insertUserBehaviorInfo(reqDo, currentDate, behaviorId);
                        } else {
                            throw new BaseBusinessException(UbcRspConstants.RSP_CODE_SYSTEM_EXCEPTION, ex.getMessage());
                        }
                    }
                } else {
                    throw new BaseBusinessException(UbcRspConstants.RSP_CODE_SYSTEM_EXCEPTION, e.getMessage());
                }
            }

        } else {
            //如果是挂载用户操作，需要根据表名更新下操作时间
            UbcUserBehaviorInfo updatePo = new UbcUserBehaviorInfo();
            updatePo.setBehaviorId(behaviorId);
            if (useServerTime) {
                updatePo.setOperateTime(currentDate);
            }else {
                updatePo.setOperateTime(reqDo.getOperateTime());
            }
            updatePo.setTableName(TableUtil.getUerBehaviorTableName(ubcProperties.getUserBehaviorInfoPrefix(), behaviorId));
            ubcUserBehaviorInfoMapper.updateOperateTimeByTableName(updatePo);
        }

        //查询用户行为操作基础属性定义
//        Map<String, String> basePropertyMap = new HashMap<>();
//        if (ObjectUtil.isNotEmpty(reqDo.getOperateInfoList())) {
//            List<UbcBehaviorOperateBaseProperty> ubcBehaviorOperateBaseProperties = ubcBehaviorOperateBasePropertyMapper.selectList(null);
//            if (ObjectUtil.isNotEmpty(ubcBehaviorOperateBaseProperties)) {
//                basePropertyMap = ubcBehaviorOperateBaseProperties.stream().collect(Collectors.toMap(UbcBehaviorOperateBaseProperty::getPropertyCode, UbcBehaviorOperateBaseProperty::getPropertyName));
//            }
//        }


        //将用户操作挂载到行为下即可
        for (UbcUserOperateInfoDo operateInfoDo : reqDo.getOperateInfoList()) {
            //查询用户行为操作定义
            UbcBehaviorOperateDefine ubcBehaviorOperateDefine = ubcBehaviorOperateDefineMapper.selectOne(new LambdaQueryWrapper<UbcBehaviorOperateDefine>().eq(UbcBehaviorOperateDefine::getOperateCode, operateInfoDo.getOperateCode()));
            if (ObjectUtil.isEmpty(ubcBehaviorOperateDefine)) {
                throw new BaseBusinessException(UbcRspConstants.RSP_CODE_BEHAVIOR_OPERATE_NOT_DEFINE, "用户行为操作编码[" + operateInfoDo.getOperateCode() + "]未定义");
            }

            //前端传入的keys
            Set<String> keySet = JSONObject.parseObject(operateInfoDo.getUserOperateData()).keySet();
            //查询用户行为操作属性定义
            List<UbcBehaviorOperateProperty> ubcBehaviorOperateProperties = ubcBehaviorOperatePropertyMapper.selectList(new LambdaQueryWrapper<UbcBehaviorOperateProperty>().eq(UbcBehaviorOperateProperty::getOperateCode, operateInfoDo.getOperateCode()));
            if (ObjectUtil.isNotEmpty(ubcBehaviorOperateProperties)) {
                Map<String, String> propertyMap = ubcBehaviorOperateProperties.stream().collect(Collectors.toMap(UbcBehaviorOperateProperty::getPropertyCode, UbcBehaviorOperateProperty::getPropertyName));
                //校验后端配置的操作编码，前端是否都传入了，没传入，报错
                for (String innerKey : propertyMap.keySet()) {
                    //如果传入的key，不能覆盖内部定义的所有key，则返回报错
                    if (!keySet.contains(innerKey)) {
                        throw new BaseBusinessException(UbcRspConstants.RSP_CODE_BEHAVIOR_OPERATE_PROPERTY_NOT_INPUT, "用户行为操作属性[" + innerKey + "]不能为空");
                    }
                }
                //合并map
//                propertyMap.putAll(basePropertyMap);
            }
            try {
                insertUserOperateInfo(reqDo, currentDate, behaviorId, operateInfoDo, ubcBehaviorOperateDefine);
            } catch (Exception e){
                //如果表不存在的情况下，先创建表，然后再插入数据
                if (e.getMessage().contains("Unknown error 1146") || e.getMessage().contains("Unknown database")) {
                    try {
                        createDatabase(dataSourceUrl, dataSourceUserName, dataSourcePassword, null, LocalDate.now().getYear(), ubcProperties.getUserOperateInfoPrefix());
                        //创建表成功后，插入数据
                        insertUserOperateInfo(reqDo, currentDate, behaviorId, operateInfoDo, ubcBehaviorOperateDefine);
                    } catch (Exception ex) {
                        //并发下，防止数据丢失
                        if (ex.getMessage().contains("Unknown error 1050") || ex.getMessage().contains("already exists")) {
                            insertUserOperateInfo(reqDo, currentDate, behaviorId, operateInfoDo, ubcBehaviorOperateDefine);
                        } else {
                            throw new BaseBusinessException(UbcRspConstants.RSP_CODE_SYSTEM_EXCEPTION, ex.getMessage());
                        }
                    }
                } else {
                    throw new BaseBusinessException(UbcRspConstants.RSP_CODE_SYSTEM_EXCEPTION, e.getMessage());
                }
            }
        }

        return reqDo;
    }

    private void insertUserOperateInfo(UbcUserBehaviorInfoDo reqDo, Date currentDate, Long behaviorId, UbcUserOperateInfoDo operateInfoDo, UbcBehaviorOperateDefine ubcBehaviorOperateDefine) {
        //组装po字段
        UbcUserOperateInfo operatePo = new UbcUserOperateInfo();
        BeanUtils.copyProperties(operateInfoDo, operatePo);
        operatePo.setOperateId(IdUtil.nextId());
        operatePo.setBehaviorId(behaviorId);
        operatePo.setUserId(reqDo.getUserId());
        operatePo.setUserName(reqDo.getUserName());
        operatePo.setBusiStart(ubcBehaviorOperateDefine.getBusiStart());
//            operatePo.setUserOperateData("data");
//            operatePo.setTmpCode(operateInfoDo.getTmpCode());

        //替换展位符
        operatePo.setOperateContent(replacePlaceHolder(operateInfoDo, ubcBehaviorOperateDefine.getOperateTemp()));
        if (useServerTime) {
            operatePo.setOperateTime(currentDate);
            operatePo.setReceiveTime(currentDate);
        }else {
            operatePo.setReceiveTime(operatePo.getOperateTime());
        }

        //用户操作，和用户行为使用不同的表前缀，但表名的年份后缀，必须使用用户行为的id，这样才能准确的根据用户行为id定位下属所有用户操作数据
        //缺陷：不能根据用户操作id，查询到用户操作的数据（目前没有这种诉求，后续如果有，则需要维护一张关联表，以操作id，作为分表的年份后缀，即可根据操作id反向查询到所属的行为id）
        operatePo.setTableName(TableUtil.getUerBehaviorTableName(ubcProperties.getUserOperateInfoPrefix(), behaviorId));
        System.out.println(JSON.toJSONString(operatePo));
        ubcUserOperateInfoMapper.insertByTableName(operatePo);
    }

    private void insertUserBehaviorInfo(UbcUserBehaviorInfoDo reqDo, Date currentDate, Long behaviorId) {
        UbcUserBehaviorInfo ubcBehaviorPo = new UbcUserBehaviorInfo();
        BeanUtils.copyProperties(reqDo, ubcBehaviorPo);
        ubcBehaviorPo.setTableName(TableUtil.getUerBehaviorTableName(ubcProperties.getUserBehaviorInfoPrefix(), behaviorId));
        if (useServerTime) {
            ubcBehaviorPo.setOperateTime(currentDate);
            ubcBehaviorPo.setReceiveTime(currentDate);
        }else {
            ubcBehaviorPo.setReceiveTime(ubcBehaviorPo.getOperateTime());
        }
        //保存用户行为
        ubcUserBehaviorInfoMapper.insertByTableName(ubcBehaviorPo);
    }

    private String replacePlaceHolder(UbcUserOperateInfoDo operateInfoDo, String operateTemp) {
        //如果不为空，才进行替换
        if (ObjectUtil.isNotEmpty(operateTemp)) {
            JSONObject jsonObject = JSONObject.parseObject(operateInfoDo.getUserOperateData());
            Properties properties = new Properties();
            for (String key : jsonObject.keySet()) {
                properties.setProperty(key, jsonObject.getString(key));
            }
            PropertyPlaceholderHelper helper = new PropertyPlaceholderHelper("${", "}");
            return helper.replacePlaceholders(operateTemp, properties);
        } else {
            return "";
        }
    }

    @Override
    public UbcUserBehaviorInfoDo qryUserBehaviorInfoDo(UbcUserBehaviorInfoDo reqDo) {
        Long behaviorId = reqDo.getBehaviorId();
        UbcUserBehaviorInfo qryPo = new UbcUserBehaviorInfo();
        qryPo.setBehaviorId(behaviorId);
        qryPo.setTableName(TableUtil.getUerBehaviorTableName(ubcProperties.getUserBehaviorInfoPrefix(), behaviorId));
        //查询用户行为
        List<UbcUserBehaviorInfo> ubcUserBehaviorInfos = ubcUserBehaviorInfoMapper.selectListByBehaviorId(qryPo);
        UbcUserBehaviorInfoDo rspDo = new UbcUserBehaviorInfoDo();
        BeanUtils.copyProperties(ubcUserBehaviorInfos.get(0), rspDo);
        //翻译业务功能
        LambdaQueryWrapper<UbcBusiDefine> queryWrapper = new LambdaQueryWrapper<>();
        queryWrapper.eq(UbcBusiDefine::getBusiCode, rspDo.getBusiCode());
        UbcBusiDefine ubcBusiDefine = ubcBusiDefineMapper.selectOne(queryWrapper);
        if (ObjectUtil.isNotEmpty(ubcBusiDefine)) {
            rspDo.setBusiDesc(ubcBusiDefine.getBusiDesc());
        }
        //根据用户行为id，查询用户操作列表
        UbcUserOperateInfo qryOperatePo = new UbcUserOperateInfo();
        qryOperatePo.setBehaviorId(behaviorId);
        qryOperatePo.setTableName(TableUtil.getUerBehaviorTableName(ubcProperties.getUserOperateInfoPrefix(), behaviorId));
        List<UbcUserOperateInfo> ubcUserOperateInfos = ubcUserOperateInfoMapper.selectListByBehaviorId(qryOperatePo);
        //用户操作详情列表，按时间逆序
        rspDo.setOperateInfoList(JUtil.jsl(ubcUserOperateInfos, UbcUserOperateInfoDo.class));

        //content按时间顺序展示
        List<UbcUserOperateInfo> contentList = new ArrayList<>(ubcUserOperateInfos);
        Collections.reverse(contentList);
        StringBuilder sb = new StringBuilder();
        for (UbcUserOperateInfo ubcUserOperateInfo : contentList) {
            sb.append(ubcUserOperateInfo.getOperateContent()).append(",");
        }
        String tmpStr = sb.toString();
        rspDo.setBehaviorContent(tmpStr.substring(0, tmpStr.length() - 1));
        return rspDo;
    }

    private void validateSnapTempCode(String snapTempCode) {
        LambdaQueryWrapper<UbcRelSnapTempType> qryWrapper = new LambdaQueryWrapper<>();
        qryWrapper.eq(UbcRelSnapTempType::getSnapTempCode, snapTempCode);
        //根据快照编码，查询配置表，如果没有配置数据，则不能保存，并返回异常
        UbcRelSnapTempType ubcRelSnapTempType = ubcRelSnapTempTypeMapper.selectOne(qryWrapper);
        if (ObjectUtil.isEmpty(ubcRelSnapTempType)) {
            throw new BaseBusinessException(UbcRspConstants.RSP_CODE_SNAPSHOT_NOT_EXIST, "快照编码[" + snapTempCode + "]存在");
        }
    }

    @Override
    public List<UbcBusiDefineBo> qryBusiDefine(UbcBusiDefineBo ubcBusiDefineBo) {
        LambdaQueryWrapper<UbcBusiDefine> wrapper = new LambdaQueryWrapper<>();
        if (ObjectUtil.isNotEmpty(ubcBusiDefineBo.getBusiCode())) {
            wrapper.eq(UbcBusiDefine::getBusiCode, ubcBusiDefineBo.getBusiCode());
        }
        if (ObjectUtil.isNotEmpty(ubcBusiDefineBo.getBusiDesc())) {
            wrapper.like(UbcBusiDefine::getBusiDesc, ubcBusiDefineBo.getBusiDesc());
        }
        return UbcRu.jsl(ubcBusiDefineMapper.selectList(wrapper), UbcBusiDefineBo.class);
    }

    @Override
    public List<UbcLocationDefineBo> qryLocationDefine(UbcLocationDefineBo ubcLocationDefineBo) {
        LambdaQueryWrapper<UbcLocationDefine> wrapper = new LambdaQueryWrapper<>();
        if (ObjectUtil.isNotEmpty(ubcLocationDefineBo.getBusiCode())) {
            wrapper.eq(UbcLocationDefine::getBusiCode, ubcLocationDefineBo.getBusiCode());
        }
        if (ObjectUtil.isNotEmpty(ubcLocationDefineBo.getLocationCode())) {
            wrapper.eq(UbcLocationDefine::getLocationCode, ubcLocationDefineBo.getLocationCode());
        }
        if (ObjectUtil.isNotEmpty(ubcLocationDefineBo.getLocationName())) {
            wrapper.like(UbcLocationDefine::getLocationName, ubcLocationDefineBo.getLocationName());
        }
        return UbcRu.jsl(ubcLocationDefineMapper.selectList(wrapper), UbcLocationDefineBo.class);
    }


    //默认翻页条数
    @Value("${sync.syncPageSize:10}")
    private long syncPageSize;
    //默认起始
    @Value("${sync.syncStart:1}")
    private long syncStart;
    @Autowired
    private UbcElasticsearchUtil ubcElasticsearchUtil;

    @Override
    public UbcSyncEsServiceRspBo syncEs(UbcSyncEsServiceReqBo reqBo) {
        //不为空，同步单条
        if (ObjectUtil.isNotEmpty(reqBo.getBehaviorId())) {
            //使用线程池，同步数据到es
            ubcThreadPool.executeRunnable(new SyncUserBehaviorToEsRunnable(reqBo.getBehaviorId()));
        } else {
            int year = LocalDate.now().getYear();
            if (ObjectUtil.isNotEmpty(reqBo.getYear())) {
                year = reqBo.getYear();
            }

            if (reqBo.isDelIndex()) {
                //删除索引，创建索引
                String indexName = ubcProperties.getUserBehaviorInfoPrefix() + TableUtil.TABLE_UNDERSCORE + year;
                ubcElasticsearchUtil.deleteIndex(indexName);
                ubcElasticsearchUtil.createIndex(indexName);
            }

            //循环查询所有数据
            long currentPage = syncStart;
            boolean hasNextPage = true;
            while (hasNextPage) {
                Page<UbcUserBehaviorInfo> page = new Page<>(currentPage, syncPageSize);
                Page<UbcUserBehaviorInfo> rspPage = qryDataAndSyncEs(page, year);
                // 当前页码，是否有下一页
                currentPage = rspPage.getCurrent();
                hasNextPage = rspPage.hasNext();
                if (!hasNextPage) {
                    break;
                }
                currentPage++;
            }
        }
        return UbcRu.success(UbcSyncEsServiceRspBo.class);
    }


    /**
     * 描述:查询并使用线程池同步es
     *
     * @param page 翻页参数
     * @param year 年份
     * @return com.baomidou.mybatisplus.extension.plugins.pagination.Page<com.tydic.dyc.ubc.repository.po.UbcUserBehaviorInfo>
     * @author tgy
     * @date 2023/8/24 18:00
     */
    private Page<UbcUserBehaviorInfo> qryDataAndSyncEs(Page<UbcUserBehaviorInfo> page, int year) {
        UbcUserBehaviorInfo qryPo = new UbcUserBehaviorInfo();
        qryPo.setTableName(ubcProperties.getUserBehaviorInfoPrefix() + TableUtil.TABLE_UNDERSCORE + year);
        Page<UbcUserBehaviorInfo> rspPage = ubcUserBehaviorInfoMapper.selectIdListByPage(page, qryPo);
        List<UbcUserBehaviorInfo> behaviorList = rspPage.getRecords();
        //放入线程池执行
        ubcThreadPool.executeRunnable(new SyncUserBehaviorToEsRunnable(behaviorList.stream().map(UbcUserBehaviorInfo::getBehaviorId).collect(Collectors.toList())));
        return rspPage;
    }
}


