package com.tydic.dyc.act.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.tydic.dyc.act.constants.DycActivityConstants;
import com.tydic.dyc.act.constants.DycActivityRspConstants;
import com.tydic.dyc.act.model.api.DycActSkuInfoModel;
import com.tydic.dyc.act.model.api.DycActSkuPoolModel;
import com.tydic.dyc.act.model.bo.ActRelSkuPoolReqBO;
import com.tydic.dyc.act.model.bo.ActRelSourcePoolReqBO;
import com.tydic.dyc.act.model.bo.ActSkuInfoBO;
import com.tydic.dyc.act.model.bo.ActSkuInfoReqBO;
import com.tydic.dyc.act.service.api.DycActSkuPoolRelService;
import com.tydic.dyc.act.service.bo.DycActSkuPoolRelReqBO;
import com.tydic.dyc.act.service.bo.DycActSkuPoolRelRspBO;
import com.tydic.dyc.act.utils.ActRu;
import com.tydic.dyc.base.exception.BaseBusinessException;
import lombok.extern.ohaotian.HTServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 类名： DycActSkuPoolReLServiceImpl
 * 说明：商品池关联商品服务实现
 *
 * @author： lishiqing
 * 时间： 2024/4/15 20:36
 */
@HTServiceImpl
public class DycActSkuPoolRelServiceImpl implements DycActSkuPoolRelService {

    @Autowired
    private DycActSkuInfoModel dycActSkuInfoModel;

    @Autowired
    private DycActSkuPoolModel dycActSkuPoolModel;

    @Override
    public DycActSkuPoolRelRspBO relSkuPool(DycActSkuPoolRelReqBO reqBO) {
        this.valData(reqBO);
        // 供应商
        switch (reqBO.getPoolType()) {
            case 1 :
            case 3 :
                supplierAndCatalog(reqBO);
                break;
            case 2:
                commodity(reqBO);
                break;
            default:
                throw new BaseBusinessException(DycActivityRspConstants.RESP_CODE_ERROR,"商品池关联商品服务入参关联类型不符合规则");
        }
        return ActRu.success(DycActSkuPoolRelRspBO.class);
    }

    private void supplierAndCatalog(DycActSkuPoolRelReqBO reqBO) {
        ActSkuInfoReqBO skuInfoReqBO = new ActSkuInfoReqBO();
        if (DycActivityConstants.PoolType.SUPPLIER.equals(reqBO.getPoolType())) {
            skuInfoReqBO.setSupplierIdList(reqBO.getRelIds());
        }
        if (DycActivityConstants.PoolType.CATALOG.equals(reqBO.getPoolType())) {
            skuInfoReqBO.setGuideIdList(reqBO.getRelIds());
        }
        List<ActSkuInfoBO> data = dycActSkuInfoModel.queryActSkuInfoList(skuInfoReqBO).getData();
        if (DycActivityConstants.RelPoolOperType.REL.equals(reqBO.getOperType())) {
            batchInsertRelSourcePool(reqBO.getRelIds(), reqBO.getPoolId());
            if (!CollectionUtil.isEmpty(data)) {
                batchInsertRelSkuPool(data.stream().map(ActSkuInfoBO::getSkuId).collect(Collectors.toList()), reqBO.getPoolId());
            }
        }
        if (DycActivityConstants.RelPoolOperType.CANCEL_REL.equals(reqBO.getOperType())) {
            batchDeleteRelSourcePool(reqBO.getRelIds(), reqBO.getPoolId());
            if (!CollectionUtil.isEmpty(data)) {
                batchDeleteRelSkuPool(data.stream().map(ActSkuInfoBO::getSkuId).collect(Collectors.toList()), reqBO.getPoolId());
            }
        }
    }

    private void commodity(DycActSkuPoolRelReqBO reqBO) {
        if (DycActivityConstants.RelPoolOperType.REL.equals(reqBO.getOperType())) {
            batchInsertRelSkuPool(reqBO.getRelIds(), reqBO.getPoolId());
        }
        if (DycActivityConstants.RelPoolOperType.CANCEL_REL.equals(reqBO.getOperType())) {
            batchDeleteRelSkuPool(reqBO.getRelIds(), reqBO.getPoolId());
        }
    }

    /**
     * 批量新增业务表
     * @param sourceIds
     * @param poolId
     */
    private void batchInsertRelSourcePool(List<Long> sourceIds,Long poolId) {
        List<ActRelSourcePoolReqBO> actRelSourcePoolReqBO = new ArrayList<>();
        for (Long sourceId : sourceIds) {
            ActRelSourcePoolReqBO relSourcePoolReqBO = new ActRelSourcePoolReqBO();
            relSourcePoolReqBO.setPoolId(poolId);
            relSourcePoolReqBO.setSourceId(sourceId);
            actRelSourcePoolReqBO.add(relSourcePoolReqBO);
        }
        dycActSkuPoolModel.batchInsertActRelSourcePool(actRelSourcePoolReqBO);
    }

    /**
     * 批量新增关联表
     * @param skuIds
     * @param poolId
     */
    private void batchInsertRelSkuPool(List<Long> skuIds,Long poolId) {
        List<ActRelSkuPoolReqBO> actRelSkuPoolReqBO = new ArrayList<>();
        for (Long skuId : skuIds) {
            ActRelSkuPoolReqBO relSkuPoolReqBO = new ActRelSkuPoolReqBO();
            relSkuPoolReqBO.setPoolId(poolId);
            relSkuPoolReqBO.setSkuId(skuId);
            actRelSkuPoolReqBO.add(relSkuPoolReqBO);
        }
        dycActSkuPoolModel.batchInsertActRelSkuPool(actRelSkuPoolReqBO);
    }

    /**
     * 批量删除关联表
     * @param skuIds
     * @param poolId
     */
    private void batchDeleteRelSkuPool(List<Long> skuIds,Long poolId) {
        ActRelSkuPoolReqBO skuPoolReqBO = new ActRelSkuPoolReqBO();
        skuPoolReqBO.setSkuIdList(skuIds);
        skuPoolReqBO.setPoolId(poolId);
        dycActSkuPoolModel.batchDeleteActRelSkuPool(skuPoolReqBO);
    }

    /**
     * 批量删除业务表
     * @param sourceIds
     * @param poolId
     */
    private void batchDeleteRelSourcePool(List<Long> sourceIds,Long poolId) {
        ActRelSourcePoolReqBO sourcePoolReqBO = new ActRelSourcePoolReqBO();
        sourcePoolReqBO.setSourceIdList(sourceIds);
        sourcePoolReqBO.setPoolId(poolId);
        dycActSkuPoolModel.batchDeleteActRelSourcePool(sourcePoolReqBO);
    }

    /**
     * 参数校验
     * @param reqBO
     */
    private void valData(DycActSkuPoolRelReqBO reqBO) {
        if (ObjectUtil.isEmpty(reqBO)) {
            throw new BaseBusinessException(DycActivityRspConstants.RSP_CODE_ACT_EMPTY_ERROR,"商品池关联商品服务入参为空");
        }
        if (ObjectUtil.isEmpty(reqBO.getPoolId())) {
            throw new BaseBusinessException(DycActivityRspConstants.RSP_CODE_ACT_EMPTY_ERROR,"商品池关联商品服务入参商品池ID[poolId]为空");
        }
        if (ObjectUtil.isEmpty(reqBO.getPoolType())) {
            throw new BaseBusinessException(DycActivityRspConstants.RSP_CODE_ACT_EMPTY_ERROR,"商品池关联商品服务入参商品池关联类型[poolType]为空");
        }
        if (ObjectUtil.isEmpty(reqBO.getRelIds())) {
            throw new BaseBusinessException(DycActivityRspConstants.RSP_CODE_ACT_EMPTY_ERROR,"商品池关联商品服务入参关联ID集合[relIds]为空");
        }
        if (ObjectUtil.isEmpty(reqBO.getOperType())) {
            throw new BaseBusinessException(DycActivityRspConstants.RSP_CODE_ACT_EMPTY_ERROR,"商品池关联商品服务入参操作类型[operType]为空");
        }
    }
}
