package com.tydic.dyc.oc.transactionservice;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.oc.constants.UocDicConstant;
import com.tydic.dyc.oc.constants.UocRspConstants;
import com.tydic.dyc.oc.constants.UocStateConstants;
import com.tydic.dyc.oc.model.audit.IUocAuditOrderModel;
import com.tydic.dyc.oc.model.audit.UocAuditOrderDo;
import com.tydic.dyc.oc.model.audit.qrybo.UocApprovalObjQryBo;
import com.tydic.dyc.oc.model.audit.qrybo.UocAuditOrderQryBo;
import com.tydic.dyc.oc.model.audit.sub.UocApprovalObj;
import com.tydic.dyc.oc.model.saleorder.IUocSaleOrderModel;
import com.tydic.dyc.oc.model.saleorder.UocSaleOrderDo;
import com.tydic.dyc.oc.service.domainservice.bo.UocAuditOrderCancelServiceReqBo;
import com.tydic.dyc.oc.service.domainservice.bo.UocAuditOrderCancelServiceRspBo;
import com.tydic.dyc.oc.utils.UocRu;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.stream.Collectors;

/**
 * 标题:UocAuditOrderCancelServiceTransaction
 * 说明:
 * 时间:2023/9/20 16:06
 * 作者:田桂银
 */
@Component
@Slf4j
public class UocAuditOrderCancelServiceTransaction {

    @Autowired
    private IUocAuditOrderModel iUocAuditOrderModel;

    @Autowired
    private IUocSaleOrderModel iUocSaleOrderModel;
    
    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public UocAuditOrderCancelServiceRspBo cancelAuditOrder(UocAuditOrderCancelServiceReqBo reqBo) {
        UocAuditOrderCancelServiceRspBo rspBo = UocRu.success(UocAuditOrderCancelServiceRspBo.class);
        // 入参校验
        validateArg(reqBo);

        // 根据销售单id，查询所属审批单数据
        UocApprovalObjQryBo qryAuditObjcBo = new UocApprovalObjQryBo();
        qryAuditObjcBo.setOrderId(reqBo.getOrderId());
        qryAuditObjcBo.setObjId(reqBo.getSaleOrderId().toString());
        log.info("取消审批单，查询审批对象列表入参：{}", JSON.toJSONString(qryAuditObjcBo));
        List<UocApprovalObj> uocApprovalObjs = iUocAuditOrderModel.qryApprovealObj(qryAuditObjcBo);
        log.info("取消审批单，查询审批对象列表出参：{}", JSON.toJSONString(uocApprovalObjs));
        if (ObjectUtil.isNotEmpty(uocApprovalObjs)) {
            Long auditOrderId = uocApprovalObjs.get(0).getAuditOrderId();
            UocAuditOrderQryBo qryAuditBo = new UocAuditOrderQryBo();
            qryAuditBo.setOrderId(reqBo.getOrderId());
            qryAuditBo.setAuditOrderId(auditOrderId);
            UocAuditOrderDo uocAuditOrderDo = iUocAuditOrderModel.qryAuditOrder(qryAuditBo);
            //判断审批单是否已经取消了，如果已经是取消状态，直接返回（支持幂等）
            if (UocStateConstants.AUDIT_PLUS_ORDER_STATUS.REVOKE.equals(uocAuditOrderDo.getAuditOrderStatus())) {
                log.info("取消审批单，已取消，返回取消成功");
                rspBo.setOrderId(reqBo.getOrderId());
                rspBo.setAuditOrderId(auditOrderId);
                rspBo.setAuditOrderNo(uocAuditOrderDo.getAuditOrderCode());
                rspBo.setCancelSuccess(true);
                return rspBo;
            } else if (UocStateConstants.AUDIT_PLUS_ORDER_STATUS.AUDITED.equals(uocAuditOrderDo.getAuditOrderStatus())) {
                //已审批，则取消失败
                log.info("取消审批单，已审批，返回取消失败");
                rspBo.setOrderId(reqBo.getOrderId());
                rspBo.setAuditOrderId(auditOrderId);
                rspBo.setAuditOrderNo(uocAuditOrderDo.getAuditOrderCode());
                rspBo.setCancelSuccess(false);
                return rspBo;
            }

            // 轮训查询的审批单下的销售单数据，判断是否有处于审批中的销售单，如果有，则取消审批单失败
            List<Long> saleOrderIdList = uocApprovalObjs.stream().map(item -> Long.parseLong(item.getObjId())).distinct().collect(Collectors.toList());
            UocSaleOrderDo qrySaleOrderListBo = new UocSaleOrderDo();
            qrySaleOrderListBo.setOrderId(reqBo.getOrderId());
            qrySaleOrderListBo.setSaleOrderIdList(saleOrderIdList);
            log.info("取消审批单，查询销售单入参：{}", JSON.toJSONString(qrySaleOrderListBo));
            List<UocSaleOrderDo> uocSaleOrderDos = iUocSaleOrderModel.qrySaleOrderList(qrySaleOrderListBo);
            log.info("取消审批单，查询销售单出参：{}", JSON.toJSONString(uocSaleOrderDos));
            if (ObjectUtil.isNotEmpty(uocSaleOrderDos)) {
                //审批单是否可以取消，默认true，一旦有一个销售单处于审批中，则不能取消
                boolean canCancel = true;
                for (UocSaleOrderDo uocSaleOrderDo : uocSaleOrderDos) {
                    if (UocStateConstants.SaleOrder.XS_SP_SPZ.equals(uocSaleOrderDo.getSaleOrderState())) {
                        canCancel = false;
                        break;
                    }
                }
                // 如果没有审批中的销售单，则直接取消审批单（更新状态）
                if (canCancel) {
                    //取消审批单
                    UocAuditOrderDo updateBo = new UocAuditOrderDo();
                    updateBo.setOrderId(reqBo.getOrderId());
                    updateBo.setAuditOrderId(auditOrderId);
                    updateBo.setAuditOrderStatus(UocStateConstants.AUDIT_PLUS_ORDER_STATUS.REVOKE);
                    log.info("取消审批单，入参：{}", JSON.toJSONString(updateBo));
                    iUocAuditOrderModel.updateApprove(updateBo);

                    rspBo.setOrderId(reqBo.getOrderId());
                    rspBo.setAuditOrderId(auditOrderId);
                    rspBo.setAuditOrderNo(uocAuditOrderDo.getAuditOrderCode());
                    rspBo.setCancelSuccess(true);
                } else {
                    log.info("取消审批单，有销售单处于审批中，不能取消审批单");
                    rspBo.setOrderId(reqBo.getOrderId());
                    rspBo.setAuditOrderId(auditOrderId);
                    rspBo.setAuditOrderNo(uocAuditOrderDo.getAuditOrderCode());
                    rspBo.setCancelSuccess(false);
                }
            }
        } else {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "审批对象数据不存在");
        }

        return rspBo;
    }

    private void validateArg(UocAuditOrderCancelServiceReqBo reqBo) {
        if (ObjectUtil.isEmpty(reqBo)) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象不能为空");
        }
        if (ObjectUtil.isEmpty(reqBo.getOrderId())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参属性orderId不能为空");
        }
        if (ObjectUtil.isEmpty(reqBo.getSaleOrderId())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象属性saleOrderId不能为空");
        }

    }

}


