package com.tydic.dyc.oc.components.thread;

import cn.hutool.core.util.ObjectUtil;
import com.tydic.dyc.base.constants.DycBaseLogConstant;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.oc.components.thread.runnable.LogTraceIdRunnable;
import com.tydic.dyc.oc.constants.UocRspConstants;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.beans.factory.config.AutowireCapableBeanFactory;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.util.concurrent.*;

/**
 * 标题:UocThreadPool
 * 说明:统一线程池
 * 时间:2023/8/21 18:13
 * 作者:田桂银
 */
@Component
public class UocThreadPool {

    private ThreadPoolExecutor executor = null;

    // 核心线程数
    @Value("${uoc.thread.corePoolSize:4}")
    int corePoolSize;

    // 最大线程数
    @Value("${uoc.thread.maximumPoolSize:50}")
    int maximumPoolSize;

    // 线程空闲时间
    @Value("${uoc.thread.keepAliveTime:60}")
    long keepAliveTime;

    /**
     * 等待队列数量
     */
    @Value("${uoc.thread.blockingCapacity:100}")
    private int blockingCapacity;

    @Autowired
    private AutowireCapableBeanFactory beanFactory;

    @PostConstruct
    public void initThreadPoolExecutor() {
        executor = new DycThreadPoolExecutor(
                corePoolSize,
                maximumPoolSize,
                keepAliveTime,
                TimeUnit.SECONDS,
                new ArrayBlockingQueue<>(blockingCapacity),
                Executors.defaultThreadFactory(),
                new ThreadPoolExecutor.AbortPolicy()
        );
    }


    private ThreadPoolExecutor getExecutor() {
        return executor;
    }

    /**
     * 描述:对外部传入Runnable对象，进行属性注入，并执行线程
     *
     * @param runnable 入参
     * @return void
     * @author tgy
     * @date 2023/8/21 18:40
     */
    public void executeRunnable(Runnable runnable) {
        if(runnable == null){
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "使用线程池创建线程时，入参Runnable对象不能为空");
        }
        //将Runnable对象，放入Spring容器中，为其属性注入值
        beanFactory.autowireBean(runnable);
        //包装成可以跟启动线程打印相同日志调用traceId的线程来运行
//        executor.execute(wrapWithLogTraceIdRunnable(runnable));
        //普通运行线程，无法打印跟启动线程相同的日志traceId（新的使用 DycThreadPoolExecutor 继承了ThreadPoolExecutor，并重写了execute()方法，包装了日志traceId）
        executor.execute(runnable);
    }

//    private Runnable wrapWithLogTraceIdRunnable(Runnable runnable) {
//        String logTraceId = MDC.get(DycBaseLogConstant.X_B3_TraceId);
//        if (ObjectUtil.isNotEmpty(logTraceId)) {
//            return new LogTraceIdRunnable(runnable, logTraceId);
//        } else {
//            return runnable;
//        }
//    }

}


