package com.tydic.dyc.oc.transactionservice;

import cn.hutool.core.util.ObjectUtil;
import com.tydic.dyc.base.constants.BaseRspConstant;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.oc.constants.UocRspConstants;
import com.tydic.dyc.oc.constants.UocStateConstants;
import com.tydic.dyc.oc.model.common.IUocCommonModel;
import com.tydic.dyc.oc.model.common.UocCommonDo;
import com.tydic.dyc.oc.model.saleorder.IUocSaleOrderModel;
import com.tydic.dyc.oc.model.saleorder.UocSaleOrderDo;
import com.tydic.dyc.oc.model.shiporder.IUocShipOrderModel;
import com.tydic.dyc.oc.model.shiporder.UocShipOrderDo;
import com.tydic.dyc.oc.model.shiporder.qrybo.UocShipOrderQryBo;
import com.tydic.dyc.oc.service.domainservice.bo.UocConfirmReceiptShipOrderBO;
import com.tydic.dyc.oc.service.domainservice.bo.UocShipOrderConfirmReceiptReqBo;
import com.tydic.dyc.oc.service.domainservice.bo.UocShipOrderConfirmReceiptRspBo;
import com.tydic.dyc.oc.service.saleorder.bo.UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo;
import com.tydic.dyc.oc.utils.UocRu;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.List;

/**
 * 描述：发货单确认收货实现类
 */
@Slf4j
@Component
public class UocShipOrderConfirmReceiptTransaction {

	@Autowired
	private IUocCommonModel iUocCommonModel;

	/**
	 * 销售单模型层
	 */
	@Autowired
	private IUocSaleOrderModel iUocSaleOrderModel;

	/**
	 * 发货单模型层
	 */
	private IUocShipOrderModel iUocShipOrderModel;

	/**
	 * 发货单确认收货
	 */
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public UocShipOrderConfirmReceiptRspBo dealConfirmReceipt(UocShipOrderConfirmReceiptReqBo reqBo) {
		//定义出参对象
		UocShipOrderConfirmReceiptRspBo rspBo = new UocShipOrderConfirmReceiptRspBo();

		//参数校验
		this.validateParam(reqBo);

		//修改销售单状态和环节编码
		this.modifySaleOrderState(reqBo);

		//提交任务
		this.submitTask(reqBo);

		//获取发货单列表
		List<UocShipOrderDo> shipOrderList = this.getShipOrderList(reqBo);

		//出参
		rspBo.setUocConfirmReceiptShipOrderBOList(UocRu.jsl(shipOrderList, UocConfirmReceiptShipOrderBO.class));
		rspBo.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
		rspBo.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
		return rspBo;
	}

	/**
	 * 参数校验
	 */
	private void validateParam(UocShipOrderConfirmReceiptReqBo reqBo) {
        if (ObjectUtil.isEmpty(reqBo)) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象不能为空");
        }
		if (ObjectUtil.isEmpty(reqBo.getTaskId())) {
			throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参[任务ID]不能为空");
		}
		if (ObjectUtil.isEmpty(reqBo.getUserId())) {
			throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参用户ID为空");
		}
	}

	/**
	 * 修改销售单状态和环节编码
	 */
	private void modifySaleOrderState(UocShipOrderConfirmReceiptReqBo reqBo) {
		//定义更新销售单状态和环节编码参数
		UocSaleOrderDo uocSaleOrderDo = new UocSaleOrderDo();
		//设置订单ID
		uocSaleOrderDo.setOrderId(reqBo.getOrderId());
		//设置销售单ID
		uocSaleOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
		//设置销售单状态
		uocSaleOrderDo.setSaleOrderState(UocStateConstants.SaleOrder.XS_QR_QRSH);
		//设置修改人ID
		uocSaleOrderDo.setUpdateOperId(String.valueOf(reqBo.getUserId()));

		//调用销售单模型，修改销售单状态和环节编码
		this.iUocSaleOrderModel.updateSaleStatusAndProCode(uocSaleOrderDo);
	}

	/**
	 * 提交任务
	 */
	private void submitTask(UocShipOrderConfirmReceiptReqBo reqBo){
		//定义订单公用DO
		UocCommonDo uocCommonDo = new UocCommonDo();

		//定义提交任务信息集合
		List<UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo> taskInfoList = new ArrayList<>(1);

		//定义任务信息
		UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo taskInfo = new UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo();
		//设置任务ID
		taskInfo.setTaskId(reqBo.getTaskId());
		//添加任务信息
		taskInfoList.add(taskInfo);

		//设置提交任务信息
		uocCommonDo.setCompleteTaskInfos(taskInfoList);
		//设置用户ID
		uocCommonDo.setUserId(reqBo.getUserId());
		//设置订单ID
		uocCommonDo.setOrderId(reqBo.getOrderId());

		//调用订单公共模型，提交任务
		this.iUocCommonModel.dealTask(uocCommonDo);
	}

	/**
	 * 获取发货单列表
	 */
	private List<UocShipOrderDo> getShipOrderList(UocShipOrderConfirmReceiptReqBo reqBo) {
		//定义查询发货单参数
		UocShipOrderQryBo uocShipOrderQryBo = new UocShipOrderQryBo();
		//设置销售单ID
		uocShipOrderQryBo.setSaleOrderId(reqBo.getSaleOrderId());
		//设置订单ID
		uocShipOrderQryBo.setOrderId(reqBo.getOrderId());

		//调用发货单模型，获取发货单列表
		return this.iUocShipOrderModel.getListShipOrder(uocShipOrderQryBo);
	}

}
