package com.tydic.dyc.oc.transactionservice;

import cn.hutool.core.util.ObjectUtil;
import com.tydic.dyc.base.constants.BaseRspConstant;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.oc.constants.*;
import com.tydic.dyc.oc.model.aforder.IUocAfOrderModel;
import com.tydic.dyc.oc.model.aforder.qrybo.UocAfOrderObjQryBo;
import com.tydic.dyc.oc.model.aforder.qrybo.UocAfOrderQryBo;
import com.tydic.dyc.oc.model.aforder.sub.UocAfOrder;
import com.tydic.dyc.oc.model.aforder.sub.UocAfOrderObj;
import com.tydic.dyc.oc.model.common.IUocCommonModel;
import com.tydic.dyc.oc.model.common.UocCommonDo;
import com.tydic.dyc.oc.model.insporder.IUocInspOrderModel;
import com.tydic.dyc.oc.model.insporder.UocInspOrderDo;
import com.tydic.dyc.oc.model.insporder.sub.UocInspOrderItem;
import com.tydic.dyc.oc.model.insporder.sub.UocShipInspRel;
import com.tydic.dyc.oc.model.order.IUocOrderModel;
import com.tydic.dyc.oc.model.order.qrybo.UocConfEffectiveQryBo;
import com.tydic.dyc.oc.model.order.sub.UocConfEffective;
import com.tydic.dyc.oc.model.order.sub.UocOrderAccessory;
import com.tydic.dyc.oc.model.order.sub.UocOrderRel;
import com.tydic.dyc.oc.model.saleorder.IUocSaleOrderModel;
import com.tydic.dyc.oc.model.saleorder.UocSaleOrderDo;
import com.tydic.dyc.oc.model.saleorder.qrybo.UocSaleOrderItemQryBo;
import com.tydic.dyc.oc.model.saleorder.qrybo.UocSaleOrderQryBo;
import com.tydic.dyc.oc.model.saleorder.sub.UocAlertInfo;
import com.tydic.dyc.oc.model.saleorder.sub.UocSaleOrderItem;
import com.tydic.dyc.oc.model.shiporder.IUocShipOrderModel;
import com.tydic.dyc.oc.model.shiporder.UocShipOrderDo;
import com.tydic.dyc.oc.model.shiporder.qrybo.UocShipOrderItemQryBo;
import com.tydic.dyc.oc.model.shiporder.qrybo.UocShipOrderQryBo;
import com.tydic.dyc.oc.model.shiporder.sub.UocShipOrderItem;
import com.tydic.dyc.oc.service.domainservice.bo.UocArrvalAcceptanceReqBo;
import com.tydic.dyc.oc.service.domainservice.bo.UocArrvalAcceptanceRspBo;
import com.tydic.dyc.oc.service.domainservice.bo.UocCreateInspOrderReqShipBo;
import com.tydic.dyc.oc.service.saleorder.bo.UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo;
import com.tydic.dyc.oc.utils.IdUtil;
import com.tydic.dyc.oc.utils.UocRu;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 描述: 〈领域服务-到货验收实现〉
 * @author : yuhao
 * @date : 2022/8/26 16:44
 */
@Slf4j
@Component
public class UocArrvalAcceptanceTransaction {

    @Autowired
    private IUocInspOrderModel iUocInspOrderModel;
    @Autowired
    private IUocShipOrderModel iUocShipOrderModel;
    @Autowired
    private IUocSaleOrderModel iUocSaleOrderModel;
    @Autowired
    private IUocOrderModel iUocOrderModel;
    @Autowired
    private IUocCommonModel iUocCommonModel;
    @Autowired
    private IUocAfOrderModel iUocAfOrderModel;
    /**
     * 全部验收完成
     */
    private static final String OP_FLAG_YES = "1";
    /**
     * 未全部验收完成
     */
    private static final String OP_FLAG_NO = "0";
    /**
     * 订单来源-电商
     */
    private static final String EC_IMPORT = "2";
    /**
     * 全部-供应商ID
     */
    private static final String SUP_NO = "0";


    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public UocArrvalAcceptanceRspBo dealArrvalAcceptance(UocArrvalAcceptanceReqBo reqBo) {

        //参数校验
        this.verifyParam(reqBo);
        //查询销售单取到是否跳过发货，到货确认标识
        UocSaleOrderQryBo uocSaleOrderQryBo = new UocSaleOrderQryBo();
        uocSaleOrderQryBo.setOrderId(reqBo.getOrderId());
        uocSaleOrderQryBo.setSaleOrderId(reqBo.getSaleOrderId());
        UocSaleOrderDo uocSaleOrderDo = this.iUocSaleOrderModel.qrySaleOrder(uocSaleOrderQryBo);
        // 是否有未完结的售后
        this.validateAfOrderUnfinished(reqBo);
        //查询发货单明细
        List<UocShipOrderItem> shipOrderItemInfos = this.getShipOrderItemInfos(reqBo);
        //查询发货单明细对应的销售单明细
        List<UocSaleOrderItem> saleOrderItemInfos = this.getSaleOrderItemInfos(reqBo, shipOrderItemInfos.stream().map(UocShipOrderItem::getSaleOrderItemId).distinct().collect(Collectors.toList()));
        //更新发货单信息
        Map<Long, Boolean> shipOrderInspStateMap = this.updateShipOrderInfo(reqBo, shipOrderItemInfos, saleOrderItemInfos);
        //更新销售单信息
        this.updateSaleOrderInfo(reqBo,shipOrderItemInfos);
        //创建验收单
        Long inspOrderId = this.createInspOrder(reqBo, shipOrderItemInfos,saleOrderItemInfos);
        //创建订单结算关联数据
        this.createOrderSettleRelInfo(reqBo,inspOrderId);
        //提交发货单任务
        if (0 == uocSaleOrderDo.getIsAgrIgnoreArrive()) {
            this.submitShipOrderTask(reqBo, shipOrderInspStateMap);
        }
        //判断销售单是否验收完成
        Boolean inspFinish = this.judgeSaleOrderFinishFlag(reqBo);
        if(inspFinish){
            //更新销售单的状态
            this.updateSaleOrderState(reqBo);
            //提交销售单任务
            if (0 == uocSaleOrderDo.getIsAgrIgnoreArrive()) {
                this.submitSaleOrderTask(reqBo);
            }
        }
        //返回结果
        UocArrvalAcceptanceRspBo rspBo = UocRu.success(UocArrvalAcceptanceRspBo.class);
        rspBo.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
        rspBo.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
        rspBo.setFinish(inspFinish);
        rspBo.setOpFlag(inspFinish ? OP_FLAG_YES : OP_FLAG_NO);
        rspBo.setInspOrderId(inspOrderId);
        rspBo.setShipOrderInspStateMap(shipOrderInspStateMap);
        rspBo.setIsAgrIgnoreArrive(uocSaleOrderDo.getIsAgrIgnoreArrive());
        return rspBo;
    }

    /**
     * 提交销售单任务
     * @param reqBo
     */
    private void submitSaleOrderTask(UocArrvalAcceptanceReqBo reqBo){
        UocCommonDo uocCommonDo = new UocCommonDo();
        List<UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo> taskBos = new ArrayList<>();
        UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo taskBo = new UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo();
        taskBo.setTaskId(reqBo.getTaskId());
        taskBo.setStepId(reqBo.getStepId());
        taskBos.add(taskBo);
        uocCommonDo.setCompleteTaskInfos(taskBos);
        uocCommonDo.setUserId(reqBo.getUserId());
        uocCommonDo.setOrderId(reqBo.getOrderId());
        this.iUocCommonModel.dealTask(uocCommonDo);
    }

    /**
     * 提交发货单任务
     * @param reqBo
     */
    private void submitShipOrderTask(UocArrvalAcceptanceReqBo reqBo,Map<Long, Boolean> shipOrderInspStateMap){
        UocCommonDo uocCommonDo = new UocCommonDo();
        List<UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo> taskBos = new ArrayList<>();
        for(UocCreateInspOrderReqShipBo uocCreateInspOrderReqShipBo:reqBo.getShipOrderList()){
            if(shipOrderInspStateMap.containsKey(uocCreateInspOrderReqShipBo.getShipOrderId()) && shipOrderInspStateMap.get(uocCreateInspOrderReqShipBo.getShipOrderId())){
                UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo taskBo = new UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo();
                taskBo.setTaskId(uocCreateInspOrderReqShipBo.getTaskId());
                taskBo.setStepId(uocCreateInspOrderReqShipBo.getStepId());
                taskBos.add(taskBo);
            }
        }
        uocCommonDo.setCompleteTaskInfos(taskBos);
        uocCommonDo.setUserId(reqBo.getUserId());
        uocCommonDo.setOrderId(reqBo.getOrderId());
        this.iUocCommonModel.dealTask(uocCommonDo);
    }


    /**
     * 创建验收单
     * @param reqBo
     * @param shipOrderItemInfos
     */
    private Long createInspOrder(UocArrvalAcceptanceReqBo reqBo,List<UocShipOrderItem> shipOrderItemInfos,List<UocSaleOrderItem> saleOrderItems){
        //将发货单明细转化为map
        Map<Long, UocShipOrderItem> uocShipOrderItemMap = shipOrderItemInfos.stream().collect(Collectors.toMap(UocShipOrderItem::getShipOrderItemId, o -> o));
    
        Map<Long, UocSaleOrderItem> uocSaleOrderItemMap = saleOrderItems.stream().collect(Collectors.toMap(UocSaleOrderItem::getSaleOrderItemId, o -> o));
        //组装验收单数据
        UocInspOrderDo createInspOrderDo = new UocInspOrderDo();
        //生产验收单id
        Long inspOrderId = IdUtil.nextId();
        //设置订单id
        createInspOrderDo.setOrderId(reqBo.getOrderId());
        //设置销售单id
        createInspOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
        //设置验收单id
        createInspOrderDo.setInspOrderId(inspOrderId);
        //设置备注
        createInspOrderDo.setRemark(reqBo.getRemark());
        //设置其他备注
        createInspOrderDo.setOtherRemark(reqBo.getOtherRemark());
        //设置验收单编号
        createInspOrderDo.setInspOrderNo(getInspOrderNo(reqBo));
        //设置验收单状态
        createInspOrderDo.setInspOrderState(UocStateConstants.InspOrder.YS_YS_YS);
        //设置验收人名称
        createInspOrderDo.setInspOper(reqBo.getName());
        //设置创建人id
        createInspOrderDo.setCreateOperId(String.valueOf(reqBo.getUserId()));
        //设置创建时间
        createInspOrderDo.setCreateTime(new Date());
        //设置开票标识
        createInspOrderDo.setInvoiceTag(UocConstant.INVOICE_TAG.NO);
        //设置验收人电话
        createInspOrderDo.setInspOperPhone(reqBo.getCellphone());
        //设置租户ID
        createInspOrderDo.setTenantId(reqBo.getTenantId());
        //设置采购金额
//        createInspOrderDo.setTotalPurchaseFee(BigDecimal.ZERO);
        //设置销售金额
//        createInspOrderDo.setTotalSaleFee(BigDecimal.ZERO);
        //定义需要创建的发货验收关系集合
        List<UocShipInspRel> uocShipInspRelList = new ArrayList<>();
        //定义需要创建的验收明细集合
        List<UocInspOrderItem> uocInspOrderItemList = new ArrayList<>();

        //组装发货验收关系
        reqBo.getShipOrderList().forEach(uocCreateInspOrderReqShipBo -> {
            UocShipInspRel createShipInspRel = new UocShipInspRel();
            //设置数据id
            createShipInspRel.setId(IdUtil.nextId());
            //设置发货单id
            createShipInspRel.setShipOrderId(uocCreateInspOrderReqShipBo.getShipOrderId());
            //设置验收单id
            createShipInspRel.setInspOrderId(createInspOrderDo.getInspOrderId());
            //设置订单id
            createShipInspRel.setOrderId(reqBo.getOrderId());
            //设置销售单id
            createShipInspRel.setSaleOrderId(reqBo.getSaleOrderId());
            //设置创建人id
            createShipInspRel.setCreateOperId(String.valueOf(reqBo.getUserId()));
            //设置创建时间
            createShipInspRel.setCreateTime(new Date());
            uocShipInspRelList.add(createShipInspRel);

            //组装验收明细
            uocCreateInspOrderReqShipBo.getShipItemList().forEach(uocCreateInspOrderReqShipItemBo -> {
                UocShipOrderItem uocShipOrderItem = uocShipOrderItemMap.get(uocCreateInspOrderReqShipItemBo.getShipItemId());
                UocInspOrderItem createInspOrderItem = new UocInspOrderItem();
                //设置订单id
                createInspOrderItem.setOrderId(reqBo.getOrderId());
                //设置销售单id
                createInspOrderItem.setSaleOrderId(reqBo.getSaleOrderId());
                //设置验收单id
                createInspOrderItem.setInspOrderId(createInspOrderDo.getInspOrderId());
                //设置验收单明细id
                createInspOrderItem.setInspOrderItemId(IdUtil.nextId());
                //设置验收数量
                createInspOrderItem.setInspCount(uocCreateInspOrderReqShipItemBo.getInspCount());
                //设置剩余验收数量
                createInspOrderItem.setRemainInspCount(uocCreateInspOrderReqShipItemBo.getInspCount());
                //设置初始验收数量
                createInspOrderItem.setInitInspCount(uocCreateInspOrderReqShipItemBo.getInspCount());
                //设置销售单明细id
                createInspOrderItem.setSaleOrderItemId(uocShipOrderItem.getSaleOrderItemId());
                //设置发货单明细id
                createInspOrderItem.setShipOrderItemId(uocCreateInspOrderReqShipItemBo.getShipItemId());
                //设置计量单位
                createInspOrderItem.setUnitName(uocShipOrderItem.getUnitName());
                //设置创建时间
                createInspOrderItem.setCreateTime(new Date());
                //设置创建人id
                createInspOrderItem.setCreateOperId(String.valueOf(reqBo.getUserId()));
                //初始化变更中的数量
                createInspOrderItem.setChngingCount(BigDecimal.ZERO);
                uocInspOrderItemList.add(createInspOrderItem);
            });
        });
        //设置验收发货关系
        createInspOrderDo.setUocShipInspRelList(uocShipInspRelList);
        //设置验收明细
        createInspOrderDo.setUocInspOrderItemList(uocInspOrderItemList);
        //设置允许变更时限
        createInspOrderDo.setAllowChngTime(this.getInspAllowChngTime(reqBo));


        //组装附件信息
        List<UocOrderAccessory> uocOrderAccessoryList = new ArrayList<>();
        if(!CollectionUtils.isEmpty(reqBo.getOrderAccessoryBoList())){
            reqBo.getOrderAccessoryBoList().forEach(obj->{
                UocOrderAccessory uocOrderAccessory = new UocOrderAccessory();
                uocOrderAccessory.setId(IdUtil.nextId());
                uocOrderAccessory.setOrderId(reqBo.getOrderId());
                uocOrderAccessory.setCreateTime(new Date());
                uocOrderAccessory.setObjType(UocDicConstant.OBJ_TYPE.INSPECTION);
                uocOrderAccessory.setObjId(inspOrderId);
                uocOrderAccessory.setAttachmentType(UocAttachementTypeConstants.ACCEPTANCE);
                uocOrderAccessory.setAccessoryUrl(obj.getAccessoryUrl());
                uocOrderAccessory.setAccessoryName(obj.getAccessoryName());
                uocOrderAccessoryList.add(uocOrderAccessory);
            });
        }
        createInspOrderDo.setUocOrderAccessoryList(uocOrderAccessoryList);
    
        //计算验收销售/采购金额
        BigDecimal inspSaleMoney = BigDecimal.ZERO;
        BigDecimal inspPurchaseMoney = BigDecimal.ZERO;
        for (UocInspOrderItem uocInspOrderItem: uocInspOrderItemList){
            
            UocSaleOrderItem saleOrderItem = uocSaleOrderItemMap.get(uocInspOrderItem.getSaleOrderItemId());
            if(null != saleOrderItem.getSalePrice() && null != saleOrderItem.getPurchaseCount()){
                BigDecimal inspSalFee = saleOrderItem.getSalePrice().multiply(uocInspOrderItem.getInspCount());
                inspSaleMoney = inspSaleMoney.add(inspSalFee);
            }
            if(null != saleOrderItem.getPurchasePrice() && null != saleOrderItem.getPurchaseCount()){
                BigDecimal purSalFee = saleOrderItem.getPurchasePrice().multiply(uocInspOrderItem.getInspCount());
        
                inspPurchaseMoney = inspPurchaseMoney.add(purSalFee);
            }
        }
        createInspOrderDo.setTotalSaleFee(inspSaleMoney);
        createInspOrderDo.setTotalPurchaseFee(inspPurchaseMoney);
        //创建验收单
        this.iUocInspOrderModel.createInspOrder(createInspOrderDo);

        return inspOrderId;
    }

    /**
     * 更新销售单信息
     * @param reqBo
     * @param shipOrderItemInfos
     */
    private void updateSaleOrderInfo(UocArrvalAcceptanceReqBo reqBo,List<UocShipOrderItem> shipOrderItemInfos){
        //将发货单明细转化为map
        Map<Long, UocShipOrderItem> uocShipOrderItemMap = shipOrderItemInfos.stream().collect(Collectors.toMap(UocShipOrderItem::getShipOrderItemId, o -> o));
        UocSaleOrderDo updateSaleOrderDo = new UocSaleOrderDo();
        //设置订单id
        updateSaleOrderDo.setOrderId(reqBo.getOrderId());
        //设置销售单id
        updateSaleOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
        //设置更新人id
        updateSaleOrderDo.setUpdateOperId(String.valueOf(reqBo.getUserId()));
        //设置更新人名称
        updateSaleOrderDo.setUpdateOperName(reqBo.getName());
        //设置更新时间
        updateSaleOrderDo.setUpdateTime(new Date());
        //组装需要更新的销售单明细
        Map<Long, UocSaleOrderItem> saleItemIdMap = new HashMap<>();
        reqBo.getShipOrderList().forEach(uocCreateInspOrderReqShipBo -> uocCreateInspOrderReqShipBo.getShipItemList().forEach(uocCreateInspOrderReqShipItemBo -> {
            UocShipOrderItem uocShipOrderItem = uocShipOrderItemMap.get(uocCreateInspOrderReqShipItemBo.getShipItemId());
            UocSaleOrderItem uocSaleOrderItem = saleItemIdMap.get(uocShipOrderItem.getSaleOrderItemId());
            if (null != uocSaleOrderItem) {
                uocSaleOrderItem.setAcceptanceCount(uocSaleOrderItem.getAcceptanceCount().add(uocCreateInspOrderReqShipItemBo.getInspCount()));
                uocSaleOrderItem.setInspSuperCount(uocSaleOrderItem.getInspSuperCount().add(uocShipOrderItem.getInspSuperCount()));
            } else {
                uocSaleOrderItem = new UocSaleOrderItem();
                uocSaleOrderItem.setSaleOrderItemId(uocShipOrderItem.getSaleOrderItemId());
                uocSaleOrderItem.setAcceptanceCount(uocCreateInspOrderReqShipItemBo.getInspCount());
                uocSaleOrderItem.setInspSuperCount(uocShipOrderItem.getInspSuperCount());
            }
            saleItemIdMap.put(uocSaleOrderItem.getSaleOrderItemId(), uocSaleOrderItem);
        }));
        //设置销售单明细
        updateSaleOrderDo.setSaleOrderItems(new ArrayList<>(saleItemIdMap.values()));
        //更新销售单明细验收数量和超验数量
        this.iUocSaleOrderModel.modifyBatchItemInspCount(updateSaleOrderDo);

        // 更新销售单明细超验标识
        this.iUocSaleOrderModel.modifyInspSuper(updateSaleOrderDo);

        // 更新销售单关闭尾差标识
        if (reqBo.getClosePennyDiff() != null && reqBo.getClosePennyDiff()) {
            updateSaleOrderDo.setClosePennyDiff(UocConstant.CLOSE_PENNY_DIFF.YES);
            this.iUocSaleOrderModel.modifySaleOrderMain(updateSaleOrderDo);
        }
    }

    /**
     * 更新销售单状态，如发货单存在拒收或者有部分验收则销售单为部分验收，全部验收为已验收
     */
    private void updateSaleOrderState(UocArrvalAcceptanceReqBo reqBo) {
        //查询销售单相关的发货单
        UocShipOrderQryBo uocShipOrderQryBo = new UocShipOrderQryBo();
        uocShipOrderQryBo.setOrderId(reqBo.getOrderId());
        uocShipOrderQryBo.setSaleOrderId(reqBo.getSaleOrderId());
        List<UocShipOrderDo> uocShipOrderDoList = this.iUocShipOrderModel.getListShipOrder(uocShipOrderQryBo);

        //组装需要更新的销售单信息
        UocSaleOrderDo updateSaleOrderDo = new UocSaleOrderDo();
        //设置订单id
        updateSaleOrderDo.setOrderId(reqBo.getOrderId());
        //设置销售单id
        updateSaleOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
        //设置更新人id
        updateSaleOrderDo.setUpdateOperId(String.valueOf(reqBo.getUserId()));
        //设置更新时间
        updateSaleOrderDo.setUpdateTime(new Date());
        //设置更新人名称
        updateSaleOrderDo.setUpdateOperName(reqBo.getName());
        //设置销售单的状态
        if (!CollectionUtils.isEmpty(uocShipOrderDoList)) {
            Set<String> shipOrderStateSet = uocShipOrderDoList.stream().map(UocShipOrderDo::getShipOrderState).collect(Collectors.toSet());
            if (shipOrderStateSet.size() == 1 && shipOrderStateSet.contains(UocStateConstants.ShipOrder.FH_FH_JS)){
                //全部拒收
                updateSaleOrderDo.setSaleOrderState(UocStateConstants.SaleOrder.XS_DH_JS);
            } else if (shipOrderStateSet.size() == 1 && shipOrderStateSet.contains(UocStateConstants.ShipOrder.FH_FH_YS)) {
                //全部验收
                updateSaleOrderDo.setSaleOrderState(UocStateConstants.SaleOrder.XS_YS_YS);
            } else if (shipOrderStateSet.contains(UocStateConstants.ShipOrder.FH_FH_YS)){
                //部分验收
                updateSaleOrderDo.setSaleOrderState(UocStateConstants.SaleOrder.XS_YS_BFYS);
            }
        }
        if (StringUtils.isNotEmpty(updateSaleOrderDo.getSaleOrderState())){
            this.iUocSaleOrderModel.updateSaleStatusAndProCode(updateSaleOrderDo);
            //更新预警数据标识
            UocAlertInfo updateUocAlertInfo = new UocAlertInfo();
            updateUocAlertInfo.setObjId(reqBo.getSaleOrderId().toString());
            updateUocAlertInfo.setObjType(UocDicConstant.OBJ_TYPE.SALE);
            updateUocAlertInfo.setAlertType(UocConstant.UocAlertType.NO_YS);
            updateUocAlertInfo.setAlertState(UocConstant.UocAlertState.DEALED);
            this.iUocSaleOrderModel.updateUocAlertInfo(updateUocAlertInfo);
        }
    }



    /**
     * 判断销售单是否验收完成
     */
    private Boolean judgeSaleOrderFinishFlag(UocArrvalAcceptanceReqBo reqBo) {
        // 如果没有选择关闭尾差，需要先校验是否发货完成
        if (reqBo.getClosePennyDiff() == null || !reqBo.getClosePennyDiff()) {
            UocSaleOrderItemQryBo uocSaleOrderItemQryBo = new UocSaleOrderItemQryBo();
            uocSaleOrderItemQryBo.setOrderId(reqBo.getOrderId());
            uocSaleOrderItemQryBo.setSaleOrderId(reqBo.getSaleOrderId());
            UocSaleOrderDo collectCountImplOrderItem = this.iUocSaleOrderModel.getCollectCountImplOrderItem(uocSaleOrderItemQryBo);
            UocSaleOrderItem uocSaleOrderItem = collectCountImplOrderItem.getSaleOrderItems().get(0);
            if (uocSaleOrderItem.getPurchaseCount().compareTo(uocSaleOrderItem.getSendCount()) != 0) {
                return false;
            }
        }
        UocShipOrderQryBo uocShipOrderQryBo = new UocShipOrderQryBo();
        uocShipOrderQryBo.setOrderId(reqBo.getOrderId());
        uocShipOrderQryBo.setSaleOrderId(reqBo.getSaleOrderId());
        List<UocShipOrderDo> shipOrderDos = this.iUocShipOrderModel.getListShipOrder(uocShipOrderQryBo);
        for (UocShipOrderDo uocShipOrderDo : shipOrderDos) {
            if (uocShipOrderDo.getFinishFlag().equals(UocConstant.SHIP_ORDER_FINISH_FLAG.NOT_FINISH)) {
                return false;
            }
        }
        return true;
    }

    /**
     * 更新发货单
     * @param reqBo
     * @param shipOrderItemInfos
     * @param saleOrderItemInfos
     */
    private Map<Long, Boolean> updateShipOrderInfo(UocArrvalAcceptanceReqBo reqBo,List<UocShipOrderItem> shipOrderItemInfos,List<UocSaleOrderItem> saleOrderItemInfos){
        //将发货单明细转化为map,方便后面运算
        Map<Long, UocShipOrderItem> uocShipOrderItemMap = shipOrderItemInfos.stream().collect(Collectors.toMap(UocShipOrderItem::getShipOrderItemId, o -> o));
        //将销售单明细转化为map,方便后面运算
        Map<Long, UocSaleOrderItem> uocSaleOrderItemMap = saleOrderItemInfos.stream().collect(Collectors.toMap(UocSaleOrderItem::getSaleOrderItemId, o -> o));
        //定义发货单验收是否完成运算结果集合
        Map<Long, Boolean> shipOrderInspStateMap = new HashMap<>();
        // 遍历入参发货单集合，运算发货单验收是否完成
        reqBo.getShipOrderList().forEach(uocCreateInspOrderReqShipBo -> {
            // 先默认已验收完成
            shipOrderInspStateMap.put(uocCreateInspOrderReqShipBo.getShipOrderId(), true);
            // 遍历发货单明细，运算发货单验收是否完成
            uocCreateInspOrderReqShipBo.getShipItemList().forEach(uocCreateInspOrderReqShipItemBo -> {
                UocShipOrderItem shipOrderItem = uocShipOrderItemMap.get(uocCreateInspOrderReqShipItemBo.getShipItemId());
                UocSaleOrderItem saleOrderItem = uocSaleOrderItemMap.get(shipOrderItem.getSaleOrderItemId());
                //获取超验比率
                BigDecimal inspectionExcessPercent = getInspectionExcessPercent(saleOrderItem.getInspectionExcessPercent());
                //计算总可验收数量 = 发货数量 + 可超验数量-已超验数量
                BigDecimal canInspCount = shipOrderItem.getSendCount().add(saleOrderItem.getPurchaseCount().multiply(inspectionExcessPercent).subtract(saleOrderItem.getInspSuperCount()));
                //计算验收数量 = 本次验收数量+已验收数量+退货数量(验收前售后退货)
                BigDecimal currentInspCount = uocCreateInspOrderReqShipItemBo.getInspCount().add(shipOrderItem.getInspCount()).add(shipOrderItem.getReturnCount());
                //判断验收数量是否大于可验收总数
                if (currentInspCount.compareTo(canInspCount) > 0) {
                    //超验会校验不通过，注释掉
                    //throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_ACCEPTANCE_ERROR, "剩余可验收数量不足，请刷新页面后重新填写验收数量");
                }
                // 判断发货单是否验收完成:如果没有选择验收完成，同时只要有一个发货单明细没有验收完成则发货单没有验收完成
                if ((reqBo.getInspFinish() == null || !reqBo.getInspFinish()) && currentInspCount.compareTo(shipOrderItem.getSendCount()) < 0) {
                    shipOrderInspStateMap.put(shipOrderItem.getShipOrderId(), false);
                }
                if (reqBo.getInspFinish()){
                    //验收完结
                    shipOrderInspStateMap.put(shipOrderItem.getShipOrderId(), true);
                }
                // 计算本次超验数量 = 验收数量-发货数量   超验 = 入参验收数量-发货 +退货+已验收
                BigDecimal currentInspSuperCount = currentInspCount.subtract(shipOrderItem.getSendCount());
                if (currentInspSuperCount.compareTo(BigDecimal.ZERO) > 0) {
                    shipOrderItem.setInspSuperCount(currentInspSuperCount);
                }else{
                    shipOrderItem.setInspSuperCount(BigDecimal.ZERO);
                }
            });
        });


        //需要更新发货单状态为已验收的发货单id集合
        List<Long> shipOrderIdList = new ArrayList<>();
        //需要更新的发货单明细集合
        List<UocShipOrderItem> shipOrderItemList = new ArrayList<>();
        reqBo.getShipOrderList().forEach(uocCreateInspOrderReqShipBo -> {
            // 筛选出已验收完成的发货单id
            if (shipOrderInspStateMap.get(uocCreateInspOrderReqShipBo.getShipOrderId())) {
                shipOrderIdList.add(uocCreateInspOrderReqShipBo.getShipOrderId());
            }
            //组装需更新的发货单明细集合
            uocCreateInspOrderReqShipBo.getShipItemList().forEach(uocCreateInspOrderReqShipItemBo -> {
                UocShipOrderItem uocShipOrderItem = new UocShipOrderItem();
                //设置发货单明细id
                uocShipOrderItem.setShipOrderItemId(uocCreateInspOrderReqShipItemBo.getShipItemId());
                //设置验收数量
                uocShipOrderItem.setInspCount(uocCreateInspOrderReqShipItemBo.getInspCount());
                //设置订单id
                uocShipOrderItem.setOrderId(reqBo.getOrderId());
                //设置销售单id
                uocShipOrderItem.setSaleOrderId(reqBo.getSaleOrderId());
                shipOrderItemList.add(uocShipOrderItem);
            });
        });

        //组织发货单需更新的数据
        UocShipOrderDo updateShipOrderDo = new UocShipOrderDo();
        //设置订单id
        updateShipOrderDo.setOrderId(reqBo.getOrderId());
        //设置销售单id
        updateShipOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
        //设置更新时间
        updateShipOrderDo.setUpdateTime(new Date());
        //设置更新人id
        updateShipOrderDo.setUpdateOperId(String.valueOf(reqBo.getUserId()));
        //设置发货单明细信息集合
        updateShipOrderDo.setShipOrderItemBoList(shipOrderItemList);

        //如果存在需要更新状态为已验收的发货单，则更新对应发货单的状态
        if (!CollectionUtils.isEmpty(shipOrderIdList)) {
            //设置需要更新状态为已验收的发货单id集合
            updateShipOrderDo.setShipOrderIdList(shipOrderIdList);
            //设置验收时间
            updateShipOrderDo.setInspTime(new Date());
            //设置发货单状态
            updateShipOrderDo.setShipOrderState(UocStateConstants.ShipOrder.FH_FH_YS);
            //设置验收人id
            updateShipOrderDo.setInspOperId(String.valueOf(reqBo.getUserId()));
            //设置验收人电话
            updateShipOrderDo.setInspOperPhone(reqBo.getCellphone());
            //设置验收人名称
            updateShipOrderDo.setInspOperName(reqBo.getName());
            //设置竣工标志
            updateShipOrderDo.setFinishFlag(UocConstant.SHIP_ORDER_FINISH_FLAG.FINISH);
            //更新发货单状态
            this.iUocShipOrderModel.modifyShipOrderMain(updateShipOrderDo);
        }

        //更新发货单明细验收信息
        this.iUocShipOrderModel.modifyShipOrderItemInspCount(updateShipOrderDo);

        return shipOrderInspStateMap;

    }



    /**
     * 获取超验比率
     * @param supInspScale
     * @return
     */
    private BigDecimal getInspectionExcessPercent(Integer supInspScale) {
        BigDecimal inspectionExcessPercent = BigDecimal.ZERO;
        if (null != supInspScale) {
            inspectionExcessPercent = new BigDecimal(supInspScale).divide(BigDecimal.valueOf(100), 2, RoundingMode.HALF_UP);
        }
        return inspectionExcessPercent;
    }


    /**
     * 获取验收单时效配置
     *
     * @param reqBo
     */
    private Date getInspAllowChngTime(UocArrvalAcceptanceReqBo reqBo) {
        //查询销售单
        UocSaleOrderQryBo saleOrderQryBo = new UocSaleOrderQryBo();
        saleOrderQryBo.setSaleOrderId(reqBo.getSaleOrderId());
        saleOrderQryBo.setOrderId(reqBo.getOrderId());
        UocSaleOrderDo saleOrderMain = this.iUocSaleOrderModel.qrySaleOrder(saleOrderQryBo);
        //查询时效类配置
        UocConfEffectiveQryBo uocConfEffectiveQryBo = new UocConfEffectiveQryBo();
        uocConfEffectiveQryBo.setEffectiveCode(UocConstant.EffectiveCode.TIME_LIMIT_FOR_ABNORMAL_ORDER_CHANGES);
        List<UocConfEffective> rows = this.iUocOrderModel.qryListPageConfEffective(uocConfEffectiveQryBo).getRows();
        Date allowChngTime = null;

        if (ObjectUtil.isNotEmpty(rows)) {
            //将时效配置转化为map
            Map<String, UocConfEffective> confEffectiveMap = rows.stream().collect(Collectors.toMap(o -> o.getBusinessTypeCode() + o.getSupplierNo(), Function.identity()));
            //找到对应配置的时效配置
            if(null != saleOrderMain){
                String key = null;
                if(null != saleOrderMain.getStakeholder()){
                    key = saleOrderMain.getOrderSource() + saleOrderMain.getStakeholder().getSupId();
                }else{
                    key = String.valueOf(saleOrderMain.getOrderSource()) + reqBo.getSupId();
                }
                if (ObjectUtil.isNotNull(confEffectiveMap.get(key))){
                    allowChngTime = new Date(confEffectiveMap.get(key).getAging());
                }else {
                    allowChngTime = new Date(confEffectiveMap.get(saleOrderMain.getOrderSource() + SUP_NO).getAging());
                }
            }else{
                allowChngTime = new Date(confEffectiveMap.get(EC_IMPORT + SUP_NO).getAging());
            }
        }

        return allowChngTime;
    }

    /**
     * 查询发货单明细对应的销售单明细
     * @param reqBo
     * @param saleOrderItemIdList
     * @return
     */
    private List<UocSaleOrderItem> getSaleOrderItemInfos(UocArrvalAcceptanceReqBo reqBo, List<Long> saleOrderItemIdList) {
        UocSaleOrderItemQryBo uocSaleOrderItemQryBo = new UocSaleOrderItemQryBo();
        //设置销售单明细id集合
        uocSaleOrderItemQryBo.setSaleOrderItemIdList(saleOrderItemIdList);
        //设置销售单id
        uocSaleOrderItemQryBo.setSaleOrderId(reqBo.getSaleOrderId());
        //设置订单id
        uocSaleOrderItemQryBo.setOrderId(reqBo.getOrderId());
        UocSaleOrderDo uocSaleOrderDo = this.iUocSaleOrderModel.qryListSaleOrderItem(uocSaleOrderItemQryBo);
        if (null != uocSaleOrderDo && !CollectionUtils.isEmpty(uocSaleOrderDo.getSaleOrderItems())) {
            return uocSaleOrderDo.getSaleOrderItems();
        }else{
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_ACCEPTANCE_ERROR, "获取销售明细为空");
        }
    }




    /**
     * 获取发货单明细
     */
    private List<UocShipOrderItem> getShipOrderItemInfos(UocArrvalAcceptanceReqBo reqBo) {
        UocShipOrderItemQryBo uocShipOrderItemQryBo = new UocShipOrderItemQryBo();
        //设置订单id
        uocShipOrderItemQryBo.setOrderId(reqBo.getOrderId());
        //设置销售单id
        uocShipOrderItemQryBo.setSaleOrderId(reqBo.getSaleOrderId());
        List<Long> shipOrderItemIdList = new ArrayList<>();
        List<Long> shipOrderIdList = new ArrayList<>();
        reqBo.getShipOrderList().forEach(uocCreateInspOrderReqShipBo -> {
            shipOrderIdList.add(uocCreateInspOrderReqShipBo.getShipOrderId());
            uocCreateInspOrderReqShipBo.getShipItemList().forEach(uocCreateInspOrderReqShipItemBo -> shipOrderItemIdList.add(uocCreateInspOrderReqShipItemBo.getShipItemId()));
        });
        //设置发货单明细id集合
        uocShipOrderItemQryBo.setShipOrderItemIdList(shipOrderItemIdList);
        //设置发货单id集合
        uocShipOrderItemQryBo.setShipOrderIdList(shipOrderIdList);
        UocShipOrderDo uocShipOrderDo = this.iUocShipOrderModel.getListShipOrderItem(uocShipOrderItemQryBo);

        Integer shipOrderItemSize = shipOrderItemIdList.size();
        if (null != uocShipOrderDo && !CollectionUtils.isEmpty(uocShipOrderDo.getShipOrderItemBoList()) && shipOrderItemSize.equals(uocShipOrderDo.getShipOrderItemBoList().size())) {
            return uocShipOrderDo.getShipOrderItemBoList();
        }else{
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_ACCEPTANCE_ERROR, "勾选的发货明细与查询结果不一致，请重新选择验收的商品信息");
        }
    }


    /**
     * 获取验收单编号
     */
    private String getInspOrderNo(UocArrvalAcceptanceReqBo reqBo) {
        UocSaleOrderDo querySaleOrderDo = new UocSaleOrderDo();
        querySaleOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
        querySaleOrderDo.setOrderId(reqBo.getOrderId());
        UocSaleOrderDo saleOrderMain = this.iUocSaleOrderModel.getSaleOrderMain(querySaleOrderDo);
        if(null != saleOrderMain){
            UocInspOrderDo uocInspOrderDo = new UocInspOrderDo();
            uocInspOrderDo.setOrderSource(saleOrderMain.getOrderSource());
            UocInspOrderDo inspOrderNo = this.iUocInspOrderModel.getInspOrderNo(uocInspOrderDo);
            if(null != inspOrderNo){
                return inspOrderNo.getInspOrderNo();
            }else{
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_ACCEPTANCE_ERROR, "生成验收单编号失败");
            }
        }else{
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_ACCEPTANCE_ERROR, "生成验收单编号失败，查询销售单数据为空");
        }
    }

    /**
     * 创建订单结算关联数据
     * @param reqBo
     * @param inspOrderId
     */
    private void createOrderSettleRelInfo(UocArrvalAcceptanceReqBo reqBo, Long inspOrderId) {
        UocSaleOrderDo querySaleOrderDo = new UocSaleOrderDo();
        querySaleOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
        querySaleOrderDo.setOrderId(reqBo.getOrderId());
        UocSaleOrderDo saleOrderMain = this.iUocSaleOrderModel.getSaleOrderMain(querySaleOrderDo);
        if(null != saleOrderMain){
            //生产服务费结算关联数据
            this.createRelInfo(reqBo, inspOrderId, UocDicConstant.REL_TYPE.SERVICE_CHARGE);
            if (UocConstant.ModelSettle.MY.equals(saleOrderMain.getModelSettle())) {
                //贸易模式：
                //生成普通开票关联数据
                this.createRelInfo(reqBo, inspOrderId, UocDicConstant.REL_TYPE.COMMON_INVOICE);
                //生成上游开票关联数据
                this.createRelInfo(reqBo, inspOrderId, UocDicConstant.REL_TYPE.UP_INVOICE);
            }else if(UocConstant.ModelSettle.CH.equals(saleOrderMain.getModelSettle())) {
                //撮合模式：
                //生成下游撮合开票关联数据
                this.createRelInfo(reqBo, inspOrderId, UocDicConstant.REL_TYPE.CH_INVOICE);
            } else {
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_ACCEPTANCE_ERROR, "生产订单结算关联数据失败：错误的结算模式");
            }
        }else{
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_ACCEPTANCE_ERROR, "生产订单结算关联数据失败：查询销售单单为空");
        }
    }

    /**
     * 生成外部关联数据
     * @param reqBo
     * @param inspOrderId
     * @param relType
     */
    private void createRelInfo(UocArrvalAcceptanceReqBo reqBo, Long inspOrderId, Integer relType) {
        UocOrderRel createOrderRel = new UocOrderRel();
        //设置数据id
        createOrderRel.setId(IdUtil.nextId());
        //设置关联状态
        createOrderRel.setRelStatus(UocDicConstant.REL_STATUS.UN_SUBMITTED);
        //设置关联类型
        createOrderRel.setRelType(relType);
        //设置订单ID
        createOrderRel.setOrderId(reqBo.getOrderId());
        //设置关联对象id
        createOrderRel.setObjId(inspOrderId);
        //设置关联对象类型
        createOrderRel.setObjType(UocDicConstant.OBJ_TYPE.INSPECTION);
        this.iUocOrderModel.createRelInfo(createOrderRel);
    }

    private void validateAfOrderUnfinished(UocArrvalAcceptanceReqBo reqBo){
        List<UocCreateInspOrderReqShipBo> shipBoList =  reqBo.getShipOrderList();

        List<UocAfOrder> afOrderList = new ArrayList<>();
        for (UocCreateInspOrderReqShipBo shipBo:shipBoList){
            UocShipOrderDo uocShipOrderDo = new UocShipOrderDo();
            uocShipOrderDo.setOrderId(reqBo.getOrderId());
            uocShipOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
            uocShipOrderDo.setShipOrderId(shipBo.getShipOrderId());
            afOrderList.addAll(getAfOrderList(uocShipOrderDo));
        }

        //UocAfOrderQryBo uocAfOrderQryBo = new UocAfOrderQryBo();
        //uocAfOrderQryBo.setSaleOrderId(reqBo.getSaleOrderId());
        //List<UocAfOrder> afOrderList = iUocAfOrderModel.getAfOrderList(uocAfOrderQryBo);
        //
        // 如果没有售后单则直接返回
        if (CollectionUtils.isEmpty(afOrderList)) return;
        // 校验未完成的订单
        afOrderList.forEach(afOrder-> {
            if (!Objects.equals(afOrder.getServState(), UocDicConstant.SERVICE_STATE.COMPLATE)
            && !Objects.equals(afOrder.getServState(), UocDicConstant.SERVICE_STATE.CANCEL)
            && !Objects.equals(afOrder.getServState(), UocDicConstant.SERVICE_STATE.APPROVAL_NOT_PASS)) {
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "该发货单中还存在未完结的售后服务单，无法进行验收操作，请在售后完结后再继续");
            }
        });
    }
    private List<UocAfOrder> getAfOrderList(UocShipOrderDo uocShipOrderDo) {
        //查询该发货单下的售后单
        UocAfOrderObjQryBo uocAfOrderObjQryBo = new UocAfOrderObjQryBo();
        uocAfOrderObjQryBo.setShipOrderId(uocShipOrderDo.getShipOrderId());
        uocAfOrderObjQryBo.setOrderId(uocShipOrderDo.getOrderId());
        uocAfOrderObjQryBo.setSaleOrderId(uocShipOrderDo.getSaleOrderId());
        List<UocAfOrderObj> afOrderObjList = iUocAfOrderModel.getAfOrderObjList(uocAfOrderObjQryBo);
        List<UocAfOrder> afOrderList = new ArrayList<>();
        if(!CollectionUtils.isEmpty(afOrderObjList)){
            UocAfOrderQryBo uocAfOrderQryBo = new UocAfOrderQryBo();
            uocAfOrderQryBo.setOrderId(uocShipOrderDo.getOrderId());
            uocAfOrderQryBo.setSaleOrderId(uocShipOrderDo.getSaleOrderId());
            uocAfOrderQryBo.setAfOrderIds(afOrderObjList.stream().map(obj->obj.getAfOrderId()).collect(Collectors.toList()));
            afOrderList = iUocAfOrderModel.getAfOrderList(uocAfOrderQryBo);
        }
        return afOrderList;
    }
    /**
     * 参数校验
     * @param reqBo
     */
    private void verifyParam(UocArrvalAcceptanceReqBo reqBo) {
        if (null == reqBo) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象不能为空");
        }
        if (null == reqBo.getOrderId()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参订单ID不能为空");
        }
        if (null == reqBo.getSaleOrderId()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参销售单ID不能为空");
        }
        if (null == reqBo.getInspFinish()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参验收完结不能为空");
        }
        if (null == reqBo.getClosePennyDiff()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参关闭尾差不能为空");
        }
        if (null == reqBo.getUserId()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参用户ID为空");
        }
        if (CollectionUtils.isEmpty(reqBo.getShipOrderList())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参发货单集合不能为空");
        }
        reqBo.getShipOrderList().forEach(uocCreateInspOrderReqShipBo -> {
            if (null == uocCreateInspOrderReqShipBo.getShipOrderId()) {
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参发货单ID不能为空");
            }
            /*if (null == uocCreateInspOrderReqShipBo.getTaskId()) {
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参任务ID不能为空");
            }*/
            /*if (null == uocCreateInspOrderReqShipBo.getStepId()) {
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参环节编码不能为空");
            }*/
            if (CollectionUtils.isEmpty(uocCreateInspOrderReqShipBo.getShipItemList())) {
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参发货明细不能为空");
            }
            uocCreateInspOrderReqShipBo.getShipItemList().forEach(uocCreateInspOrderReqShipItemBo -> {
                if (null == uocCreateInspOrderReqShipItemBo.getInspCount()) {
                    throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参验收数量不能为空");
                }
                if (null == uocCreateInspOrderReqShipItemBo.getShipItemId()) {
                    throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参发货明细ID不能为空");
                }
            });
        });
    }
}
