package com.tydic.dyc.oc.components.es;


import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.util.EntityUtils;
import org.elasticsearch.action.bulk.BulkRequest;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.update.UpdateRequest;
import org.elasticsearch.action.update.UpdateResponse;
import org.elasticsearch.client.*;
import org.elasticsearch.client.indices.CreateIndexRequest;
import org.elasticsearch.client.indices.CreateIndexResponse;
import org.elasticsearch.xcontent.XContentType;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;


/**
 * 标题：es获取客户端工具类
 * <p>
 * 说明：es获取客户端工具类
 * <br>
 * 时间：2019/1/1616:52<br>
 * </p>
 *
 * @author hegy
 */
@Slf4j
@Component
public class UocElasticsearchUtil {
	private static final Logger LOGGER = LoggerFactory.getLogger(UocElasticsearchUtil.class);

	@Autowired
	private RestHighLevelClient client;

	@Qualifier("restClientBuilder")
	@Autowired
	private RestClientBuilder restClientBuilder;

	private static RequestOptions options = RequestOptions.DEFAULT;

	static {
		// 自定义返回参数大小 单位 BYTE
		RequestOptions.Builder requestOptions = RequestOptions.DEFAULT.toBuilder();
		requestOptions.setHttpAsyncResponseConsumerFactory(new HttpAsyncResponseConsumerFactory.HeapBufferedResponseConsumerFactory(500 * 1024 * 1024));
		options = requestOptions.build();
	}

	private static ObjectMapper mapper = new ObjectMapper();

	private final static int batchSize = 1000;


	/**
	 * 创建索引
	 *
	 * @param index
	 * @return
	 */
	public boolean createIndex(String index) {
		//index名必须全小写，否则报错
		CreateIndexRequest request = new CreateIndexRequest(index);

		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
			CreateIndexResponse indexResponse = client.indices().create(request, options);
			if (indexResponse.isAcknowledged()) {
				LOGGER.info("创建索引成功");
			} else {
				LOGGER.info("创建索引失败");
			}
			return indexResponse.isAcknowledged();
		} catch (IOException e) {
			LOGGER.error("创建索引失败", e);
		}

		return false;
	}

	/**
	 * 插入数据
	 *
	 * @param index
	 * @param type
	 * @param object
	 * @return
	 */
	public String addData(String index, String type, String id, JSONObject object) {

		IndexRequest indexRequest = new IndexRequest();
		indexRequest.index(index);
		indexRequest.id(id);
		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
			indexRequest.source(mapper.writeValueAsString(object), XContentType.JSON);
			IndexResponse indexResponse = client.index(indexRequest, options);
			return indexResponse.getId();
		} catch (Exception e) {
			LOGGER.info("插入数据失败", e);
		}
		return null;
	}

	/**
	 * 插入数据
	 *
	 * @param index
	 * @param id
	 * @param jsonStr
	 * @return
	 */
	public String addJsonData(String index,String id, String jsonStr) {

		IndexRequest indexRequest = new IndexRequest();
		indexRequest.index(index);
		indexRequest.id(id);
		log.info("addJsonData保存数据：" + id + "es关键字：" + index);
		log.info("addJsonData保存数据：" + jsonStr);
		//log.info("es存储源数据：" + indexRequest.source().utf8ToString());
/*
		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
*/
		try {
			indexRequest.source(jsonStr, XContentType.JSON);
			IndexResponse indexResponse = client.index(indexRequest, options);
			log.info(id + "es关键字：" + index + "同步结果：" + JSON.toJSONString(indexResponse));
			return indexResponse.getId();
		} catch (Exception e) {
			LOGGER.info("插入数据失败", e);
		}
		return null;
	}


	/**
	 * 批量插入
	 *
	 * @param index
	 * @param type
	 * @param list
	 */
	public void addBatch(String index, String type, List<Map<String, JSONObject>> list) {
		if (list.isEmpty()) {
			return;
		}
		int pages = (int) Math.ceil(list.size() / (double) batchSize);
		for (int i = 0; i < pages; i++) {
			List<Map<String, JSONObject>> subList = subList(list, batchSize, i);
			dealBatch(index, type, subList);
			LOGGER.info("批量处理第--" + i + "次,共计:" + subList.size() + "--条...");
		}
		LOGGER.info("插入完毕，共计:" + list.size() + "--条...");
	}


	/**
	 * 循环截取某页列表进行分页
	 *
	 * @param dataList    分页数据
	 * @param pageSize    页面大小
	 * @param currentPage 当前页面
	 */
	public List<Map<String, JSONObject>> subList(List<Map<String, JSONObject>> dataList, int pageSize, int currentPage) {
		List<Map<String, JSONObject>> currentPageList = new ArrayList<>();
		if (dataList != null && dataList.size() > 0) {
			int currIdx = (currentPage > 0 ? currentPage * pageSize : 0);
			for (int i = 0; i < pageSize && i < dataList.size() - currIdx; i++) {
				Map<String, JSONObject> map = dataList.get(currIdx + i);
				currentPageList.add(map);
			}
		}
		return currentPageList;
	}

	/**
	 * 处理批量插入
	 *
	 * @param index
	 * @param type
	 * @param list
	 */
	private void dealBatch(String index, String type, List<Map<String, JSONObject>> list) {
		BulkRequest request = new BulkRequest();


		list.forEach(map -> {

			for (Map.Entry<String, JSONObject> entry : map.entrySet()) {

				IndexRequest indexRequest = new IndexRequest();
				indexRequest.index(index);
				indexRequest.id(entry.getKey());
				try {
					indexRequest.source(mapper.writeValueAsString(entry.getValue()), XContentType.JSON);
				} catch (JsonProcessingException e) {
					e.printStackTrace();
				}
				request.add(indexRequest);
			}
		});

		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
			BulkResponse responses = client.bulk(request, options);
			if (responses.hasFailures()) {
				LOGGER.info("批量插入有失败数据!");
			}

		} catch (IOException e) {
			e.printStackTrace();
			LOGGER.error("商品信息同步失败：" + e.getMessage());
		}

	}


	/**
	 * 根据条件删除ES数据
	 *
	 * @param endPoint 接口index点
	 * @param qryJson  删除数据语句
	 * @return boolean
	 */
	public Boolean deleteDataByCondition(String endPoint, JSONObject qryJson) {
		boolean deleteResult = false;
		HttpEntity entity = new NStringEntity(qryJson.toJSONString(), ContentType.APPLICATION_JSON);
		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
			Request request = new Request("POST", endPoint);
			request.setEntity(entity);
			Response response = client.getLowLevelClient().performRequest(request);
			String result = EntityUtils.toString(response.getEntity());
			if (StringUtils.isNotBlank(result)) {
				JSONObject deleteResultJson = JSONObject.parseObject(result);
				int deleteCount = deleteResultJson.getInteger("deleted");
				if (deleteCount > 0) {
					deleteResult = true;
				}
			}
		} catch (IOException e) {
			LOGGER.error("根据条件删除ES数据失败", e);
		}
		return deleteResult;
	}


	/**
	 * @param index
	 * @param type
	 * @param id
	 * @param qryJson
	 * @return
	 */
	public Boolean updateData(String index, String type, String id, JSONObject qryJson) {
		boolean deleteResult = false;
		IndexRequest indexRequest = new IndexRequest();
		indexRequest.index(index);
		indexRequest.id(id);
		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
			indexRequest.source(mapper.writeValueAsString(qryJson), XContentType.JSON);
			UpdateRequest updateRequest = new UpdateRequest();
			updateRequest.index(index);
			updateRequest.id(id);
			updateRequest.doc(indexRequest);
			log.info("updateData保存数据：" + id + "es关键字：" + index);
			log.info("updateData保存数据：" + id + "es关键字同步内容：" + JSON.toJSONString(indexRequest));
			UpdateResponse rspose = client.update(updateRequest, options);
			log.info(id + "es关键字：" + index + "同步结果：" + JSON.toJSONString(rspose));
			log.info(id + "es关键字：" + index + "同步结果：" + rspose.getResult().toString());
			if ("UPDATED".equals(rspose.getResult().toString())) {
				return true;
			}

			LOGGER.info(JSONObject.toJSONString(rspose));
		} catch (Exception e) {
			e.printStackTrace();
			LOGGER.error("es 更新异常：" + e.getMessage());
			return false;
		}
		return deleteResult;
	}

	public Boolean updateBatchData(String index, Map<Long, Map<String, Object>> maps) {
		boolean deleteResult = false;

		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
			BulkRequest request = new BulkRequest();
			maps.forEach((id, map) -> request.add(new UpdateRequest(index, id.toString()).doc(map).upsert()));
			BulkResponse response = client.bulk(request, RequestOptions.DEFAULT);
			LOGGER.info(JSONObject.toJSONString(response));

		} catch (Exception e) {
			e.printStackTrace();
			LOGGER.error("es 更新异常：" + e.getMessage());
			return false;
		}
		return deleteResult;
	}

	/**
	 * @param index
	 * @param id
	 * @return
	 */
	public Boolean deleteData(String index, String id) {
		boolean deleteResult = false;
		IndexRequest indexRequest = new IndexRequest();
		indexRequest.index(index);
		indexRequest.id(id);
		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
			DeleteRequest deleteRequest = new DeleteRequest();
			deleteRequest.index(index);
			deleteRequest.id(id);
			DeleteResponse rspose = client.delete(deleteRequest, options);
			LOGGER.info(JSONObject.toJSONString(rspose));
		} catch (Exception e) {
			e.printStackTrace();
			LOGGER.error("es 删除异常：" + e.getMessage());
			return false;
		}
		return deleteResult;
	}

	/**
	 * @param index
	 * @return
	 */
	public Boolean deleteIndex(String index) {
		boolean deleteResult = false;
		IndexRequest indexRequest = new IndexRequest();
		indexRequest.index(index);
		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
			DeleteRequest deleteRequest = new DeleteRequest();
			deleteRequest.index(index);
			DeleteResponse rspose = client.delete(deleteRequest, options);
			LOGGER.info(JSONObject.toJSONString(rspose));
		} catch (Exception e) {
			e.printStackTrace();
			LOGGER.error("es 删除异常：" + e.getMessage());
			return false;
		}
		return deleteResult;
	}

	/**
	 * @param index
	 * @param ids
	 * @return
	 */
	public Boolean deleteBatchData(String index, List<String> ids) {
		boolean deleteResult = false;
		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
			BulkRequest request = new BulkRequest();
			for (String id : ids) {
				request.add(new DeleteRequest().index(index).id(id));
			}
			BulkResponse response = client.bulk(request, RequestOptions.DEFAULT);
			LOGGER.info(JSONObject.toJSONString(response));
		} catch (Exception e) {
			e.printStackTrace();
			LOGGER.error("es 删除异常：" + e.getMessage());
			return false;
		}
		return deleteResult;
	}


	/**
	 * @param index
	 * @param type
	 * @param qryJson
	 * @return
	 */
	public String updateByQuery(String index, String type, String qryJson) {

		String endPoint = "/" + index + "/" + type + "/_update_by_query";

		if (LOGGER.isDebugEnabled()) {
			LOGGER.debug("source---->" + qryJson);
		}
		HttpEntity entity = new NStringEntity(qryJson, ContentType.APPLICATION_JSON);
		Request request = new Request("POST", endPoint);
		request.setEntity(entity);
		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
			Response response = client.getLowLevelClient().performRequest(request);

			String result = EntityUtils.toString(response.getEntity());
			return result;

		} catch (Exception e) {
			LOGGER.error(e.getMessage());
			return "";
		}
	}

	/**
	 * @param index
	 * @param type
	 * @param qryJson
	 * @return
	 */
	public String doPostSendQuery(String index, String type, String qryJson) {

		String endPoint = "/" + index + "/" + type + "/_search";

		if (LOGGER.isDebugEnabled()) {
			LOGGER.debug("source---->" + qryJson);
		}
		HttpEntity entity = new NStringEntity(qryJson, ContentType.APPLICATION_JSON);
		Request request = new Request("POST", endPoint);
		request.setEntity(entity);
		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
			Response response = client.getLowLevelClient().performRequest(request);

			String result = EntityUtils.toString(response.getEntity());
			return result;

		} catch (Exception e) {
			LOGGER.error(e.getMessage());
			return "";
		}
	}

	/**
	 * @param index 索引名称
	 * @param type  索引类型
	 * @param list  集合
	 * @param id    es 的_id
	 */
	public void bulkPutIndex(String index, String type, List<Map<String, Object>> list, String id) {

		if (CollectionUtils.isEmpty(list)) {
			return;
		}

		int size = list.size();
		BulkRequest request = new BulkRequest();
		for (int i = 0; i < size; i++) {
			Map<String, Object> map = list.get(i);
			if (!StringUtils.isEmpty(id)) {
				request.add(new IndexRequest(index, type, String.valueOf(map.get(id))).source(map, XContentType.JSON));
			} else if (map.containsKey("id")) {
				request.add(new IndexRequest(index, type, String.valueOf(map.remove("id"))).source(map, XContentType.JSON));
			} else {
				request.add(new IndexRequest(index, type, null).source(map, XContentType.JSON));
			}
		}
		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
			client.bulk(request, options);
		} catch (IOException e) {
			LOGGER.error("批量插入es 失败：" + e.getMessage());
			e.printStackTrace();
		}
	}

	/**
	 * 检查索引
	 *
	 * @param index
	 * @return
	 * @throws IOException
	 */
	public boolean checkIndexExist(String index) {
		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
			Request request = new Request("HEAD", index);
			Response response = client.getLowLevelClient().performRequest(request);
			boolean exist = response.getStatusLine().getReasonPhrase().equals("OK");
			return exist;
		} catch (IOException e) {
			LOGGER.error("检查索引失败", e);
		}
		return false;
	}


	/**
	 * 获取低水平客户端(支持ALIYun)
	 *
	 * @return
	 */
	public RestClient getLowLevelClient() {
		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {
			RestHighLevelClient restHighLevelClient = client;
			LOGGER.debug("RestHighLevelClient---->" + restHighLevelClient);
			return restHighLevelClient.getLowLevelClient();
		}catch (Exception e){
			LOGGER.debug("RestHighLevelClient---->" + "获取异常");
		}
		return null;
	}


	/**
	 * 获取索引分词内容
	 *
	 * @param index   索引
	 * @param qryJson 待分词短语
	 * @return
	 */

	public String getIndexAnalyze(String index, String qryJson) {
		String endPoint = "/" + index + "/_analyze";
		if (LOGGER.isDebugEnabled()) {
			LOGGER.debug("source---->" + qryJson);
		}
		HttpEntity entity = new NStringEntity(qryJson, ContentType.APPLICATION_JSON);
		Request request = new Request("POST", endPoint);
		request.setEntity(entity);
		try (RestHighLevelClient client = new RestHighLevelClient(restClientBuilder)) {

			Response response = client.getLowLevelClient().performRequest(request);

			String result = EntityUtils.toString(response.getEntity());
			return result;

		} catch (Exception e) {
			LOGGER.error(e.getMessage());
			return "";
		}
	}
}
