package com.tydic.dyc.oc.utils;

import com.alibaba.fastjson.JSONObject;
import com.ohaotian.plugin.common.util.DelFormatHelper;
import com.ohaotian.plugin.common.util.JsonUtils;
import com.tydic.dyc.oc.utils.bo.ESBCommParam;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.text.ParseException;
import java.util.*;

/**
 * <br>
 * 标题: ESB能力平台报文封装<br>
 * 描述: <br>
 *
 * @author QIJIANFEI
 * time 2017年8月18日 上午9:15:23
 */
public class ESBParamUtil {

    private static final Logger logger = LoggerFactory.getLogger(ESBParamUtil.class);

    private static final String SUPPLIER_ID = "supplierId";

    private static final String ENCODING = "UTF-8";
    private static String appkey;
    private static String key;

    static {
        appkey = UocOrderPropertiesUtil.getProperty("ESB_CGD_APPKEY");
        key = UocOrderPropertiesUtil.getProperty("ESB_CGD_KEY");
    }

    /**
     * <br>
     * 适用场景: 封装ESB请求参数<br>
     * 调用方式: 请求入参可以直接调用ESB时<br>
     * 业务逻辑说明<br>
     *
     * @param t      请求对象
     * @param hsn    请求类型
     * @param center 请求系统
     * @return String
     * @author QIJIANFEI
     * time 2017年8月18日 下午2:46:53
     */
    public static <T> String getEsbReqParam(T t, String hsn, String center) {

        try {
            JSONObject resultJson = new JSONObject();

            JSONObject busiParamJson = (JSONObject) JSONObject.toJSON(t);
            JSONObject esbCommParamJson = getEsbCommParam(hsn, center);
            busiParamJson.remove(SUPPLIER_ID);
            resultJson.putAll(busiParamJson);
            resultJson.putAll(esbCommParamJson);

            return JSONObject.toJSONString(resultJson);
        } catch (Exception e) {
            logger.error("ESB能力平台报文封装工具获取ESB请求报文失" + e);
            throw new RuntimeException("获取ESB请求报文失败");
        }
    }

    /**
     * <br>
     * 适用场景: 封装ESB请求参数<br>
     * 调用方式: 业务报文需要重新定义时<br>
     * 业务逻辑说明<br>
     *
     * @param reqJsonStr 请求数据
     * @param hsn        请求类型
     * @param center     请求系统
     * @return <T>
     * @author QIJIANFEI
     * time 2017年8月26日 下午6:24:04
     */
    public static <T> String getEsbReqStr(String reqJsonStr, String hsn, String center) {

        try {
            JSONObject resultJson = new JSONObject();

            JSONObject reqJsonObject = JSONObject.parseObject(reqJsonStr);
            JSONObject esbCommParamJson = getEsbCommParam(hsn, center);
            resultJson.putAll(esbCommParamJson);
            resultJson.putAll(reqJsonObject);

            return JSONObject.toJSONString(resultJson);
        } catch (Exception e) {
            logger.error("ESB能力平台报文封装工具获取ESB请求报文失" + e);
            throw new RuntimeException("获取ESB请求报文失败");
        }
    }

    /**
     * <br>
     * 适用场景: 解析响应数据,有一定的局限性，如果属性只是一个list解析后的数据不能使用list的遍历<br>
     * 调用方式: <br>
     * 业务逻辑说明<br>
     *
     * @param clazzT 响应类实体
     * @param clazzB 响应类实体中属性result类型
     * @return author QIJIANFEI
     * time 2017年8月21日 下午12:18:33
     */
    public static <T, B> T parsedEsbRspToObject(Class<T> clazzT, Class<B> clazzB, String rspStr) {
        try {
            T t = clazzT.newInstance();
            B b = clazzB.newInstance();

            List<Field> fieldArr = getDeclaredField(t.getClass(), Object.class);
            // 判断响应数据是否是list且是唯一的
            Boolean isHasList = isHasList(fieldArr);
            // 遍历所有属性
            for (Field field : fieldArr) {
                // 获取属性的名字
                String name = field.getName();
                // 将属性的首字符大写，方便构造get，set方法
                name = name.substring(0, 1).toUpperCase() + name.substring(1);
                if (!"RespCode".equals(name) && !"RespDesc".equals(name)) {
                    Class<?> fieldClass = field.getType();
                    if (isHasList) {
                        B[] bResultArr = (B[]) JsonUtils.jsonStringToJavaBean(rspStr, Object[].class);

                        List<B> bResult = Arrays.asList(bResultArr);
                        setValue(fieldClass, name, bResult, t, clazzB);
                    } else {
                        t = (T) JsonUtils.jsonStringToJavaBean(rspStr, t.getClass());
                    }
                }
            }

            return t;
        } catch (Exception e) {
            logger.error("ESB能力平台报文封装工具解析ESB响应数据失败" + e);
            throw new RuntimeException("解析ESB响应数据失败");
        }
    }

    /**
     * <br>
     * 适用场景: ESB能力平台公共参数封装<br>
     * 调用方式: <br>
     * 业务逻辑说明<br>
     *
     * @param hsn    请求类型
     * @param center 请求系统
     * @return JSONObject
     * @author QIJIANFEI
     * time 2017年8月18日 下午2:12:19
     */
    private static JSONObject getEsbCommParam(String hsn, String center) {
        try {
            ESBCommParam eSBCommParam = new ESBCommParam();
            eSBCommParam.setAppkey(appkey);
            eSBCommParam.setHsn(hsn);

            String timeStamp = getTimeStamp();
            eSBCommParam.setTimeStamp(timeStamp);

            String serialnumber = getSerialNumber(timeStamp, center);
            eSBCommParam.setSerialNumber(serialnumber);

            String token = ESBSignUtil.getESBToken(ENCODING, key, appkey, serialnumber, timeStamp);
            eSBCommParam.setToken(token);

            return (JSONObject) JSONObject.toJSON(eSBCommParam);
        } catch (Exception e) {
            logger.error("ESB能力平台报文封装工具获取ESB公共参数失败" + e);
            throw new RuntimeException("获取ESB公共参数失败" + e);
        }
    }

    /**
     * <br>
     * 适用场景: 获取当前时间 字符串格式 YYYY-MM-dd HH:mm:ss<br>
     * 调用方式: <br>
     * 业务逻辑说明<br>
     *
     * @return String
     * @author QIJIANFEI
     * time 2017年8月18日 上午11:28:00
     */
    private static String getTimeStamp() {
        return DelFormatHelper.getNowTimeForString();
    }

    /**
     * <br>
     * 适用场景:获取流水号 <br>
     * 调用方式: <br>
     * 业务逻辑说明<br>
     * <p>
     * throws ParseException
     * author QIJIANFEI
     * time 2017年8月18日 上午11:28:14
     */
    private static String getSerialNumber(String timeStamp, String center) throws ParseException {
        return UocOrderPropertiesUtil.getProperty(center) + DelFormatHelper.getNowTimeForString(timeStamp) + get4Random();
    }

    /**
     * <br>
     * 适用场景: 获取4位随机数<br>
     * 调用方式: <br>
     * 业务逻辑说明<br>
     *
     * @return String
     * @author QIJIANFEI
     * time 2017年8月18日 上午11:47:45
     */
    private static String get4Random() {
        int min = 1000;
        int max = 9999;
        Random random = new Random();

        int resultInt = random.nextInt(max) % (max - min + 1) + min;

        return String.valueOf(resultInt);
    }

    /**
     * <br>
     * 适用场景: 给实体属性赋值<br>
     * 调用方式: <br>
     * 业务逻辑说明<br>
     *
     * @param fieldClass 字段类型
     * @param name       名称
     * @param newValue   值
     * @param model      对象
     * @author QIJIANFEI
     * time 2017年8月21日 下午12:18:07
     */
    private static <T, B> void setValue(Class<?> fieldClass, String name, Object newValue, T model, Class<B> clazzB) {
        try {
            // 如果type是类类型，则前面包含"class "，后面跟类名
            if (fieldClass.isAssignableFrom(String.class)) {
                Method m = model.getClass().getMethod("get" + name);
                // 调用getter方法获取属性值
                String value = (String) m.invoke(model);
                if (value == null) {
                    m = model.getClass().getMethod("set" + name, String.class);
                    m.invoke(model, newValue);
                }
            }
            if (fieldClass.isAssignableFrom(Integer.class)) {
                Method m = model.getClass().getMethod("get" + name);
                Integer value = (Integer) m.invoke(model);
                if (value == null) {
                    m = model.getClass().getMethod("set" + name, Integer.class);
                    m.invoke(model, newValue);
                }
            }
            if (fieldClass.isAssignableFrom(Boolean.class)) {
                Method m = model.getClass().getMethod("get" + name);
                Boolean value = (Boolean) m.invoke(model);
                if (value == null) {
                    m = model.getClass().getMethod("set" + name, Boolean.class);
                    m.invoke(model, newValue);
                }
            }
            if (fieldClass.isAssignableFrom(Date.class)) {
                Method m = model.getClass().getMethod("get" + name);
                Date value = (Date) m.invoke(model);
                if (value == null) {
                    m = model.getClass().getMethod("set" + name, Date.class);
                    m.invoke(model, newValue);
                }
            }
            if (fieldClass.isAssignableFrom(List.class)) {
                Method m = model.getClass().getMethod("get" + name);
                List value = (List) m.invoke(model);
                if (value == null) {
                    m = model.getClass().getMethod("set" + name, List.class);
                    m.invoke(model, newValue);
                }
            }
            if (fieldClass.isAssignableFrom(clazzB)) {
                Method m = model.getClass().getMethod("get" + name);
                B value = (B) m.invoke(model);
                if (value == null) {
                    m = model.getClass().getMethod("set" + name, clazzB);
                    m.invoke(model, newValue);
                }
            }
        } catch (Exception e) {
            logger.error("ESB能力平台报文封装工具解析ESB响应数据给实体属性赋值失败" + e);
            throw new RuntimeException("解析ESB响应数据给实体属性赋值失败");
        }
    }

    /**
     * <br>
     * 适用场景: 获取实体类的所有属性<br>
     * 调用方式: <br>
     * 业务逻辑说明<br>
     *
     * @param c         类
     * @param stopClass 停止类
     * @return List
     * @author QIJIANFEI
     * time 2017年8月21日 上午11:40:49
     */
    private static List<Field> getDeclaredField(Class<?> c, Class<?> stopClass) {
        List<Field> list = new ArrayList<>();
        for (; c != stopClass; c = c.getSuperclass()) {
            Field fields[] = c.getDeclaredFields();
            for (Field field : fields) {
                if (!Modifier.isStatic(field.getModifiers())) {
                    list.add(field);
                }
            }
        }
        return list;
    }

    /**
     * <br>
     * 适用场景: 响应实体属性是否是list且唯一<br>
     * 调用方式: <br>
     * 业务逻辑说明<br>
     *
     * @param fieldArr 属性
     * @return author QIJIANFEI
     * time 2017年8月24日 下午6:03:14
     */
    private static Boolean isHasList(List<Field> fieldArr) {
        // 遍历所有属性
        for (Field field : fieldArr) {
            if (!"respCode".equals(field.getName()) && !"respDesc".equals(field.getName())) {
                Class fieldClazz = field.getType();
                if (!fieldClazz.isAssignableFrom(List.class)) {
                    return false;
                }
            }
        }
        return true;
    }
}
