package com.tydic.dyc.oc.transactionservice;

import cn.hutool.core.util.ObjectUtil;
import com.tydic.dyc.base.constants.BaseRspConstant;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.oc.constants.UocConstant;
import com.tydic.dyc.oc.constants.UocDicConstant;
import com.tydic.dyc.oc.constants.UocRspConstants;
import com.tydic.dyc.oc.model.common.IUocCommonModel;
import com.tydic.dyc.oc.model.common.UocCommonDo;
import com.tydic.dyc.oc.model.order.IUocOrderModel;
import com.tydic.dyc.oc.model.order.qrybo.UocOrderTaskInstQryBo;
import com.tydic.dyc.oc.model.order.sub.UocOrderTaskInst;
import com.tydic.dyc.oc.model.saleorder.IUocSaleOrderModel;
import com.tydic.dyc.oc.model.saleorder.UocSaleOrderDo;
import com.tydic.dyc.oc.model.saleorder.qrybo.UocSaleOrderQryBo;
import com.tydic.dyc.oc.model.shiporder.UocShipOrderDo;
import com.tydic.dyc.oc.service.domainservice.bo.UocPayOrderCallbackReqBo;
import com.tydic.dyc.oc.service.domainservice.bo.UocPayOrderCallbackRspBo;
import com.tydic.dyc.oc.service.saleorder.bo.UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo;
import com.tydic.dyc.oc.utils.UocRu;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 描述：订单在线支付回调信息处理接口实现类
 */
@Slf4j
@Component
public class UocPayOrderCallbackTransaction {

	@Autowired
	private IUocCommonModel iUocCommonModel;

	/**
	 * 销售单模型层
	 */
	@Autowired
	private IUocSaleOrderModel iUocSaleOrderModel;

	/**
	 * 订单模型层
	 */
	@Autowired
	private IUocOrderModel iUocOrderModel;

	/**
	 * 订单在线支付回调信息处理
	 */
	@Transactional(propagation = Propagation.REQUIRES_NEW)
	public UocPayOrderCallbackRspBo dealPayOrderCallback(UocPayOrderCallbackReqBo reqBo) {
		//定义出参对象
		UocPayOrderCallbackRspBo rspBo = new UocPayOrderCallbackRspBo();

		//参数校验
		this.validateParam(reqBo);

		//修改支付信息
		this.updatePayOrderInfo(reqBo);

		//处理在线支付销售单信息
		this.dealPaySaleOrderInfo(reqBo);

		//获取任务信息
		UocOrderTaskInst uocOrderTaskInst = this.getTaskInstList(reqBo);

		//提交任务
		this.submitTask(uocOrderTaskInst);

		//获取销售单
		UocSaleOrderDo uocSaleOrderDo = this.getUocSaleOrder(reqBo);

		//出参
		rspBo.setOrderId(uocSaleOrderDo.getSaleOrderId());
		rspBo.setSaleOrderId(uocSaleOrderDo.getSaleOrderId());
		rspBo.setTaskId(uocOrderTaskInst.getTaskInstId());
		rspBo.setProcState(uocOrderTaskInst.getProcState());
		rspBo.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
		rspBo.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
		return rspBo;
	}

	/**
	 * 参数校验
	 */
	private void validateParam(UocPayOrderCallbackReqBo reqBo) {
        if (ObjectUtil.isEmpty(reqBo)) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象不能为空");
        }
		if (ObjectUtil.isEmpty(reqBo.getPayResult())) {
			throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参[支付结果]不能为空");
		}
	}

	/**
	 * 获取销售单
	 */
	private UocSaleOrderDo getUocSaleOrder(UocPayOrderCallbackReqBo reqBo) {
		//定义销售单主体查询BO
		UocSaleOrderQryBo qryUocSaleOrderQryBo = new UocSaleOrderQryBo();
		//设置销售单ID
		qryUocSaleOrderQryBo.setSaleOrderId(reqBo.getSaleOrderId());

		//调用销售单模型，查询销售单主体
		return this.iUocSaleOrderModel.qrySaleOrder(qryUocSaleOrderQryBo);
	}

	/**
	 * 获取任务信息
	 */
	private UocOrderTaskInst getTaskInstList(UocPayOrderCallbackReqBo reqBo) {
		//定义查询任务信息对象
		UocOrderTaskInstQryBo uocOrderTaskInstQryBo = new UocOrderTaskInstQryBo();
		//设置对象ID
		uocOrderTaskInstQryBo.setObjId(reqBo.getSaleOrderId());
		//设置对象类型
		uocOrderTaskInstQryBo.setObjType(UocDicConstant.OBJ_TYPE.SALE);
		//设置完成标识
		uocOrderTaskInstQryBo.setFinishTag(UocDicConstant.PROC_TASK_FINISHED.NO_FINISHED);

		//调用订单模型层，获取任务信息
		List<UocOrderTaskInst> uocOrderTaskInstList = this.iUocOrderModel.qryOrderTaskInstList(uocOrderTaskInstQryBo);

		if (CollectionUtils.isEmpty(uocOrderTaskInstList)) {
			throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "任务不存在");
		}

		if (uocOrderTaskInstList.size() > 1) {
			throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "任务数量大于1");
		}

		return uocOrderTaskInstList.get(0);
	}

	/**
	 * 修改支付信息
	 */
	private void updatePayOrderInfo(UocPayOrderCallbackReqBo reqBo) {
		//定义销售单DO，参数转换
		UocSaleOrderDo payOrderUocSaleOrderDo = UocRu.js(reqBo, UocSaleOrderDo.class);
		//设置销售单支付状态
		payOrderUocSaleOrderDo.setPayState(UocConstant.PAY_RESULT.SUCCESS.equals(reqBo.getPayResult()) ? UocDicConstant.PAY_STATE.PAYED : UocDicConstant.PAY_STATE.PAY_FAILED);

		//调用销售单模型层，订单在线支付回调信息处理，修改销售单支付信息
		this.iUocSaleOrderModel.dealPayOrderCallback(payOrderUocSaleOrderDo);
	}

	/**
	 * 处理在线支付销售单信息
	 */
	private void dealPaySaleOrderInfo(UocPayOrderCallbackReqBo reqBo) {
		//定义销售单DO,参数转换
		UocSaleOrderDo uocSaleOrderDo = UocRu.js(reqBo, UocSaleOrderDo.class);

		//调用订单公共模型层，处理在线支付销售单信息
		this.iUocCommonModel.dealPaySaleOrderInfo(uocSaleOrderDo);
	}

	/**
	 * 提交任务
	 */
	private void submitTask(UocOrderTaskInst uocOrderTaskInst){
		if ("E0015".equals(uocOrderTaskInst.getProcState())) {
			//定义订单公用DO
			UocCommonDo uocCommonDo = new UocCommonDo();

			//定义提交任务信息集合
			List<UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo> taskInfoList = new ArrayList<>(1);

			//定义任务信息
			UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo taskInfo = new UocCreateSaleProcInsUpdateStatusDomainServiceReqTaskBo();
			//设置任务ID
			taskInfo.setTaskId(uocOrderTaskInst.getTaskInstId());
			//添加任务信息
			taskInfoList.add(taskInfo);

			//设置提交任务信息
			uocCommonDo.setCompleteTaskInfos(taskInfoList);
			//设置用户ID
			uocCommonDo.setUserId(1L);
			//设置订单ID
			uocCommonDo.setOrderId(uocOrderTaskInst.getOrderId());

			//调用订单公共模型，提交任务
			this.iUocCommonModel.dealTask(uocCommonDo);
		}
	}

}
