package com.tydic.dyc.oc.components.es;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.oc.constants.UocRspConstants;
import com.tydic.dyc.oc.utils.UocRu;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 标题:UocEsSyncManager
 * 说明:es同步管理类
 * 时间:2022/3/1 11:40
 * 作者:田桂银
 */
@Component
@Slf4j
public class UocEsSyncManager {

    /**
     * 在系统启动时，缓存数据查询实现类
     */
    private final Map<String, IUocEsSync> manageMap = new ConcurrentHashMap<>();

    @Autowired
    private UocElasticsearchUtil uocElasticsearchUtil;

    private static final String OBJ_JSON = "objJson";
    /**
     * 描述:保存数据
     *
     * @param obyType 入参
     * @return com.tydic.dyc.oc.components.es.IUocEsSync
     * @author tgy
     * @date 2022/3/1 12:50
     */
    public void saveData(UocEsSyncManagerSaveDataReqBo obyType) {
        log.debug("验收单es同步：" + JSON.toJSONString(obyType));
        String indexName = obyType.getIndexName();
        IUocEsSync interObj = manageMap.get(indexName);
        if (interObj == null) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_FAIL, "索引同步实现类（" + indexName + "）未找到实现类");
        }
        //入参校验
        validateQryReqBo(obyType);
        try{
            //数据查询
            IUocEsSyncQryRspBo qryRspBo = interObj.qryData(UocRu.js(obyType, IUocEsSyncQryReqBo.class));
            log.debug("验收单es同步：" + JSON.toJSONString(qryRspBo));
            // 如果不需要同步
            if ( !qryRspBo.getSyncFlag() ) {
                log.debug("验收单es同步无需同步：" + JSON.toJSONString(qryRspBo));
                return;
            }
            //出参校验
            validateRspArg(qryRspBo);
            JSONObject data = qryRspBo.getJsonObj();
//        data.put(OBJ_JSON, data.toJSONString());
            log.info("UocEsSyncManager保存es入参：" + data.toJSONString());
            //保存数据
            uocElasticsearchUtil.addJsonData(interObj.indexName(), qryRspBo.getObjId().toString(), data.toJSONString());
        } catch (Exception e){
            log.error("同步es报错:" + e);
            e.printStackTrace();
        }
    }

    public void deleteData(UocEsSyncManagerSaveDataReqBo uocEsSyncManagerSaveDataReqBo) {
        String indexName = uocEsSyncManagerSaveDataReqBo.getIndexName();
        //入参校验
        validateQryReqBo(uocEsSyncManagerSaveDataReqBo);
        IUocEsSync interObj = manageMap.get(indexName + "_delete");
        IUocEsSyncQryRspBo qryRspBo = interObj.qryData(UocRu.js(uocEsSyncManagerSaveDataReqBo, IUocEsSyncQryReqBo.class));
        if (null == qryRspBo || null == qryRspBo.getJsonObj()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象属性[json]对象不能为空");
        }
        String endPoint = "/" + indexName + "/_delete_by_query";
        //保存数据
        uocElasticsearchUtil.deleteDataByCondition(endPoint, qryRspBo.getJsonObj());
    }

    public Boolean updateData(UocEsSyncManagerSaveDataReqBo uocEsSyncManagerSaveDataReqBo) {
        String indexName = uocEsSyncManagerSaveDataReqBo.getIndexName();
        if (null == uocEsSyncManagerSaveDataReqBo.getJsonObj()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象属性[json]对象不能为空");
        }
        if (null == uocEsSyncManagerSaveDataReqBo.getObjId()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象ID[objId]对象不能为空");
        }
        String endPoint = "/" + indexName + "/_update_by_query?refresh=true";
        //保存数据
        //return uocElasticsearchUtil.updateDataById(endPoint, uocEsSyncManagerSaveDataReqBo.getJsonObj(), uocEsSyncManagerSaveDataReqBo.getObjId().toString());
        return uocElasticsearchUtil.updateData(indexName, null, uocEsSyncManagerSaveDataReqBo.getObjId().toString(), uocEsSyncManagerSaveDataReqBo.getJsonObj());
    }

    private void validateQryReqBo(UocEsSyncManagerSaveDataReqBo reqBo) {
        if (reqBo == null) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象[UocEsSyncManagerSaveDataReqBo]不能为空");
        }
        if (ObjectUtil.isEmpty(reqBo.getIndexName())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象属性[索引名称]不能为空");
        }
        if (ObjectUtil.isEmpty(reqBo.getOrderId())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象属性[orderId]不能为空");
        }
        if (ObjectUtil.isEmpty(reqBo.getObjId())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象属性[objId]不能为空");
        }

    }

    private void validateRspArg(IUocEsSyncQryRspBo reqBo) {
        if (reqBo == null) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象[IUocEsSyncQryRspBo]不能为空");
        }
        if (ObjectUtil.isEmpty(reqBo.getObjId())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象属性[对象id]不能为空");
        }
        if (ObjectUtil.isEmpty(reqBo.getJsonObj())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象属性[json]对象不能为空");
        }
    }


    /**
     * 描述:注册实现类
     *
     * @param intObj 入参
     * @return void
     * @author tgy
     * @date 2022/3/1 12:50
     */
    public void register(IUocEsSync intObj) {
        String indexName = intObj.indexName();

        Assert.notNull(intObj, "intObj can not be null");
        Assert.notNull(indexName, "indexName can not be null");

        //此处获取到得必须为空，不为空则为重复
        IUocEsSync intRet = manageMap.get(indexName);
        if (intRet != null) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_FAIL, "不同索引实现类编码(" + indexName + ")出现多个实现类！");
        }
        log.info(UocEsSyncManager.class.getSimpleName() + ".register注册了不同索引实现类：" + indexName);

        manageMap.put(indexName, intObj);
    }


}


