package com.tydic.picker.config;

import com.ohaotian.plugin.cache.CacheClient;
import com.tydic.picker.PickerClient;
import com.tydic.picker.advice.SyncDataByPointCutAdvice;
import com.tydic.picker.disruptor.PickerClientDisruptorClientPublisher;
import com.tydic.picker.properties.PickerProperties;
import com.tydic.picker.service.DataSyncService;
import com.tydic.picker.service.imp.DataSyncServiceImpl;
import com.tydic.picker.utils.ElasticsearchUtil;
import lombok.extern.slf4j.Slf4j;
import org.elasticsearch.client.RestHighLevelClient;
import org.springframework.aop.aspectj.AspectJExpressionPointcutAdvisor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.util.StringUtils;

/**
 * @ClassName PickerClientConfiguration
 * @Description 数据同步客户端配置
 * @Author liugs
 * @Date 2022/8/16 10:32
 */
@Slf4j
@Configuration
@ConditionalOnProperty(prefix = "picker.client", name = "enable", havingValue = "true")
@EnableConfigurationProperties(PickerProperties.class)
public class PickerClientConfiguration {
    /**
     * 切点包路径分隔符
     */
    private static final String PACKAGE_DELIMITER = ",";

    /**
     * 调用配置项
     */
    private PickerProperties pickerProperties;

    public PickerClientConfiguration(PickerProperties pickerProperties) {
        this.pickerProperties = pickerProperties;
    }

    @Bean(value = "elasticsearchUtil")
    public ElasticsearchUtil elasticsearchUtil(RestHighLevelClient client) {
        return new ElasticsearchUtil(client);
    }

    @Bean(value = "pickerClient", destroyMethod = "close")
    public PickerClient pickerClient(CacheClient cacheClient, DataSyncService dataSyncService) {
        log.info("初始化数据同步客户端");
        PickerClient pickerClient = new PickerClient();
        PickerClientDisruptorClientPublisher publisher = PickerClientDisruptorClientPublisher.getInstance();
        // 启动发布者
        publisher.start(dataSyncService);
        // 初始化客户端
        pickerClient.init(publisher, cacheClient, dataSyncService);

        return pickerClient;
    }

    @Bean
    @Order(-1)
    public AspectJExpressionPointcutAdvisor syncDataByPointCutAdvice(PickerClient pickerClient) {
        AspectJExpressionPointcutAdvisor advisor = new AspectJExpressionPointcutAdvisor();
        //组装切点规则
        StringBuilder expressionBuilder = null;
        if (StringUtils.hasText(pickerProperties.getPointcutPackage())) {
            if (pickerProperties.getPointcutPackage().contains(PACKAGE_DELIMITER)) {
                String[] nativeRpcPackageSplit = pickerProperties.getPointcutPackage().split(PACKAGE_DELIMITER);
                for (String nativeRpcPackageString : nativeRpcPackageSplit) {
                    if (StringUtils.hasText(nativeRpcPackageString)) {
                        if (null == expressionBuilder) {
                            expressionBuilder = new StringBuilder();
                            expressionBuilder.append("(execution(* ").append(nativeRpcPackageString).append("..*(..)))");
                        } else {
                            expressionBuilder.append(" || (execution(* ").append(nativeRpcPackageString).append("..*(..)))");
                        }
                    }
                }
            } else {
                expressionBuilder = new StringBuilder();
                expressionBuilder.append("(execution(* ").append(pickerProperties.getPointcutPackage()).append("..*(..)))");
            }
        }

        String expression = String.valueOf(expressionBuilder);
        advisor.setExpression(expression);
        log.info("数据同步切面表达式为：" + expression);
        advisor.setAdvice(new SyncDataByPointCutAdvice(pickerClient));
        advisor.setOrder(Integer.MAX_VALUE - 200);

        log.info("数据同步切面配置完成，切点规则为：{}", expression);
        return advisor;
    }

    @Bean
    public RecordTableCheckRunner recordTableCheckRunner() {
        return new RecordTableCheckRunner();
    }

    @Bean(value = "dataSyncService")
    public DataSyncService DataSyncService(JdbcTemplate jdbcTemplate, ElasticsearchUtil elasticsearchUtil) {
       return new DataSyncServiceImpl(jdbcTemplate, elasticsearchUtil);
    }
}
