package com.tydic.picker.service.imp;

import cn.hutool.core.codec.Base64Encoder;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.serializer.SerializerFeature;
import com.tydic.picker.constant.PickerConstants;
import com.tydic.picker.dto.DataPickDTO;
import com.tydic.picker.dto.PickerRecordDTO;
import com.tydic.picker.enums.ConstantEnum;
import com.tydic.picker.enums.ResultCodeEnum;
import com.tydic.picker.result.PickerResult;
import com.tydic.picker.service.DataSyncService;
import com.tydic.picker.utils.DynamicSqlUtil;
import com.tydic.picker.utils.ElasticsearchUtil;
import com.tydic.picker.utils.Sequence;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.jdbc.core.JdbcTemplate;

import javax.annotation.Resource;
import java.util.Map;

/**
 * @ClassName DataSyncServiceImpl
 * @Description 数据同步服务
 * @Author liugs
 * @Date 2022/8/15 16:48
 */
@Slf4j
public class DataSyncServiceImpl implements DataSyncService {

    @Resource
    private DynamicSqlUtil dynamicSqlUtil;

    private JdbcTemplate jdbcTemplate;

    private ElasticsearchUtil elasticsearchUtil;

    public DataSyncServiceImpl(JdbcTemplate jdbcTemplate, ElasticsearchUtil elasticsearchUtil) {
        this.jdbcTemplate = jdbcTemplate;
        this.elasticsearchUtil = elasticsearchUtil;
    }

    @Override
    public PickerResult doSync(DataPickDTO dto) {
        log.info("==开始采集数据并同步ES");
        PickerResult result = new PickerResult(ResultCodeEnum.SUCCESS);
        // 记录同步信息
        Long recordId = null;
        try {
            recordId = createSyncRecord(dto);
        } catch (Exception e) {
            log.error("数据同步记录入库异常：{}", e);
        }

        String errorMessage = null;
        try {
            // 执行动态SQL并根据映射配置填充ES对象
            JSONObject esData = dynamicSqlUtil.getMappedData(dto);
            log.debug("根据映射关系的到的数据对象：{}", JSON.toJSONString(esData, SerializerFeature.WriteMapNullValue));

            if (ObjectUtil.isNotEmpty(esData.get(PickerConstants.DOCUMENT_ID))) {
                switch (dto.getEventTypeEnum()) {
                    case CREATE:
                        createDoc(esData, dto);
                        break;
                    case UPDATE:
                        updateDoc(esData, dto);
                        break;
                    case DELETE:
                        deleteDoc(esData, dto);
                        break;
                    default:
                        break;
                }
            } else {
                errorMessage = String.format("根据配置的文档ID取值字段[%s]未获取到文档ID", dto.getDocIdField());
            }
        } catch (Exception e) {
            log.error("同步ES数据异常：");
            e.printStackTrace();
            errorMessage = "同步ES数据异常：" + e;
        }

        // 更新同步信息
        try {
            if (StringUtils.isEmpty(errorMessage)) {
                updateRecord(recordId, ConstantEnum.PickerRecordStatusEnum.SUCCESS, ConstantEnum.PickerRecordStatusEnum.SUCCESS.getDesc());
            } else {
                updateRecord(recordId, ConstantEnum.PickerRecordStatusEnum.FAIL, errorMessage);
            }
        } catch (Exception e) {
            log.error("数据同步记录状态更新异常：{}", e);
        }

        if (StringUtils.isNotEmpty(errorMessage)) {
            result.buildFail(errorMessage);
        }
        log.info("==采集数据并同步ES完成：{}", JSON.toJSONString(result));
        return result;
    }



    /**
     * 创建文档
     * @param esData
     * @param dto
     */
    private void createDoc(JSONObject esData, DataPickDTO dto) {
        elasticsearchUtil.addDocument(dto.getIndexName(), esData.remove(PickerConstants.DOCUMENT_ID).toString(), esData);
    }

    /**
     * 更新文档
     * @param esData
     * @param dto
     */
    private void updateDoc(JSONObject esData, DataPickDTO dto) {
        elasticsearchUtil.getDocById(dto.getIndexName(), esData.getString(PickerConstants.DOCUMENT_ID));
        elasticsearchUtil.updateDoc(dto.getIndexName(), esData.remove(PickerConstants.DOCUMENT_ID).toString(), esData);
    }

    /**
     * 删除文档
     * @param esData
     * @param dto
     */
    private void deleteDoc(JSONObject esData, DataPickDTO dto) {
        elasticsearchUtil.deleteDocById(dto.getIndexName(), esData.getString(PickerConstants.DOCUMENT_ID));
    }


    /**
     * 处理同步记录
     * @param dto
     * @return recordId
     */
    private Long createSyncRecord(DataPickDTO dto) {
        Long id = Sequence.nextId();
        PickerRecordDTO recordPo = new PickerRecordDTO();
        BeanUtils.copyProperties(dto, recordPo);
        recordPo.setId(id);
        recordPo.setConditionParam(JSON.toJSONString(dto.getConditionParam()));
        recordPo.setEventType(dto.getEventTypeEnum().getEventType());
        recordPo.setStartTime(new DateTime().toString("yyyy-MM-dd HH:mm:ss"));
        recordPo.setStatusDesc(ConstantEnum.PickerRecordStatusEnum.INIT.getDesc());
        recordPo.setDynamicSql(StringUtils.isEmpty(dto.getDynamicSql()) ? null : Base64Encoder.encode(dto.getDynamicSql()));
        String dynamicSql = ConstantEnum.DynamicSqlEnum.INSERT.getDynamicSql(JSON.parseObject(JSON.toJSONString(recordPo), Map.class));
        log.debug("记录数据同步SQL：{}", dynamicSql);
        if (1 > jdbcTemplate.update(dynamicSql)) {
            log.error("记录数据同步记录失败");
            return null;
        }
        return id;
    }

    /**
     * 更新记录状态
     * @param recordId
     * @param statusEnum
     * @param desc
     * @return
     */
    private void updateRecord(Long recordId, ConstantEnum.PickerRecordStatusEnum statusEnum, String desc) {
        PickerRecordDTO recordPo = new PickerRecordDTO();
        recordPo.setId(recordId);
        recordPo.setStatus(statusEnum.getState());
        recordPo.setStatusDesc(desc);
        recordPo.setFinishTime(new DateTime().toString("yyyy-MM-dd HH:mm:ss"));
        String dynamicSql = ConstantEnum.DynamicSqlEnum.UPDATE.getDynamicSql(JSON.parseObject(JSON.toJSONString(recordPo), Map.class));
        log.debug("记录数据同步SQL：{}", dynamicSql);
        if ( 1 > jdbcTemplate.update(dynamicSql)) {
            log.error("更新数据同步记录失败");
        }
    }
}
