package com.tydic.tanggula.advice;

import com.tydic.tanggula.TanggulaClient;
import com.tydic.tanggula.dto.TanggulaMethodParamDTO;
import com.tydic.tanggula.enums.TanggulaResultCodeEnum;
import com.tydic.tanggula.properties.TanggulaProperties;
import com.tydic.tanggula.result.TanggulaResult;
import lombok.extern.slf4j.Slf4j;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.springframework.util.ObjectUtils;

import java.lang.reflect.Field;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 标题：类名称:TanggulaLogAdvice
 * 说明：描述一下类的作用TODO
 * 时间：2024/1/29 14:46
 * 作者 @author yyd
 */
@Slf4j
public class TanggulaLogAdvice implements MethodInterceptor {

    private TanggulaClient tanggulaClient;

    private TanggulaProperties properties;

    private static final String SPLIT_SEPARATOR = ",";

    public TanggulaLogAdvice(TanggulaClient tanggulaClient, TanggulaProperties properties){
        this.tanggulaClient = tanggulaClient;
        this.properties = properties;
    }

    @Override
    public Object invoke(MethodInvocation invocation) throws Throwable {
        Map<String, Object> inputMap = new HashMap<>(16);
        Map<String, Object> outputMap = new HashMap<>(16);
        dealInArgument(invocation.getArguments(), inputMap);
        Object rspObj = invocation.proceed();
        dealObject(rspObj, outputMap);

        String logMethodsStr = properties.getLogMethods();
        List<Long> logMethods = new ArrayList<>();
        try {
            logMethods = Arrays.stream(logMethodsStr.split(SPLIT_SEPARATOR)).map(Long::valueOf).collect(Collectors.toList());
        }catch (Exception e){
            log.error("日志采集方式配置异常，请检查配置，异常信息：{}", e.getMessage());
        }

        TanggulaMethodParamDTO methodParamDTO = TanggulaMethodParamDTO.builder()
                .serviceCode(invocation.getMethod().getDeclaringClass().getName())
                .methodCode(invocation.getMethod().getName())
                .topic(properties.getTanggulaLogTopic())
                .tag(properties.getTanggulaLogTag())
                .inputMap(inputMap)
                .outputMap(outputMap)
                .logMethods(logMethods)
                .build();

        TanggulaResult result = null;
        try {
            result = tanggulaClient.execute(methodParamDTO);
        }catch (Exception e){
            log.error(e.getMessage());
        }

        if (ObjectUtils.isEmpty(result) || !TanggulaResultCodeEnum.SUCCESS.getCode().equals(result.getCode())) {
            log.error("日志记录发生异常，异常信息：{}", result.getMessage());
        }
        return rspObj;
    }

    private void dealInArgument(Object[] arguments, Map<String, Object> map) throws Throwable {
        if (arguments != null) {
            for (Object arg : arguments) {
                dealObject(arg, map);
            }
        }
    }

    private void dealObject(Object dealObject, Map<String, Object> fieldValueMap) throws Throwable {
        final List<Field> fields = getAllField(dealObject.getClass());
        for (Field f : fields) {
            //设置些属性是可以访问的
            f.setAccessible(true);
            //属性名
            String fieldName = f.getName();
            if (null != f.get(dealObject)) {
                Object fieldValue = f.get(dealObject);
                fieldValueMap.put(fieldName, fieldValue);
            }
        }
    }

    private static List<Field> getAllField(Class clazz) {
        List<Field> fields = new ArrayList<>();
        while (clazz != null) {
            fields.addAll(new ArrayList<>(Arrays.asList(clazz.getDeclaredFields())));
            clazz = clazz.getSuperclass();
        }
        return fields;
    }
}
