package com.tydic.tanggula.config;

import com.ohaotian.plugin.cache.CacheClient;
import com.ohaotian.plugin.mq.proxy.DefaultProxyMessageConfig;
import com.ohaotian.plugin.mq.proxy.ProxyMessageProducer;
import com.ohaotian.plugin.mq.proxy.RedisCacheStore;
import com.ohaotian.plugin.mq.proxy.impl.ProxyProducerFactoryBean;
import com.tydic.tanggula.TanggulaClient;
import com.tydic.tanggula.advice.TanggulaLogAdvice;
import com.tydic.tanggula.properties.TanggulaProperties;
import com.tydic.tanggula.service.impl.TanggulaGenCodeLogAbleImpl;
import com.tydic.tanggula.service.impl.TanggulaUserOperLogAbleImpl;
import com.tydic.tanggula.util.TanggulaLogCollectAbleManager;
import lombok.extern.slf4j.Slf4j;
import org.springframework.aop.aspectj.AspectJExpressionPointcutAdvisor;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.DependsOn;
import org.springframework.util.StringUtils;


/**
 * 标题：类名称:TanggulaClientConfiguration
 * 说明：描述一下类的作用TODO
 * 时间：2024/1/29 11:08
 * 作者 @author yyd
 */
@Slf4j
@Configuration
@ConditionalOnProperty(prefix = "tanggula.client", name = "enable", havingValue = "true")
@EnableConfigurationProperties(TanggulaProperties.class)
public class TanggulaClientConfiguration {

    /**
     * 切点包路径分隔符
     */
    private static final String PACKAGE_DELIMITER = ",";

    private TanggulaProperties properties;

    private CacheClient cacheClient;

    public TanggulaClientConfiguration(TanggulaProperties properties, CacheClient cacheClient){
         this.cacheClient = cacheClient;
         this.properties = properties;
    }

    @Bean(value = "pluginProvider", initMethod = "startup", destroyMethod = "shutdown")
    @ConditionalOnMissingBean(ProxyProducerFactoryBean.class)
    public ProxyProducerFactoryBean proxyProducerFactoryBean(){
        DefaultProxyMessageConfig config = new DefaultProxyMessageConfig();
        config.setId(properties.getTanggulaLogPid());
        ProxyProducerFactoryBean proxyProducerFactoryBean = new ProxyProducerFactoryBean();
        proxyProducerFactoryBean.setMessageConfig(config);
        return proxyProducerFactoryBean;
    }

    @Bean
    public RedisCacheStore redisCacheStore(){
        RedisCacheStore redisCacheStore = new RedisCacheStore();
        return redisCacheStore;
    }

    @Bean
    public TanggulaUserOperLogAbleImpl tanggulaUserOperLogAbleImpl(){
        TanggulaUserOperLogAbleImpl tanggulaUserOperLogAble = new TanggulaUserOperLogAbleImpl(cacheClient);
        return tanggulaUserOperLogAble;
    }

    @Bean
    public TanggulaGenCodeLogAbleImpl tanggulaGenCodeLogAbleImpl(){
        TanggulaGenCodeLogAbleImpl tanggulaGenCodeLogAble = new TanggulaGenCodeLogAbleImpl(cacheClient);
        return tanggulaGenCodeLogAble;
    }

    @Bean
    public TanggulaLogCollectAbleManager tanggulaLogCollectAbleManager(TanggulaUserOperLogAbleImpl tanggulaUserOperLogAble, TanggulaGenCodeLogAbleImpl tanggulaGenCodeLogAble){
        TanggulaLogCollectAbleManager tanggulaLogCollectAbleManager = new TanggulaLogCollectAbleManager();
        tanggulaLogCollectAbleManager.registerLogCollectionAble(tanggulaUserOperLogAble);
        tanggulaLogCollectAbleManager.registerLogCollectionAble(tanggulaGenCodeLogAble);
        return tanggulaLogCollectAbleManager;
    }
    @Bean(value = "tanggulaClient")
    @DependsOn("pluginProvider")
    public TanggulaClient tanggulaClient(@Qualifier("pluginProvider") ProxyMessageProducer pluginProvider,
                                         TanggulaLogCollectAbleManager tanggulaLogCollectAbleManager){
        TanggulaClient tanggulaClient = new TanggulaClient(pluginProvider, tanggulaLogCollectAbleManager);
        return tanggulaClient;
    }

    @Bean
    public AspectJExpressionPointcutAdvisor userOperateLogRecordAdvisor(TanggulaClient tanggulaClient){
        AspectJExpressionPointcutAdvisor advisor = new AspectJExpressionPointcutAdvisor();

        //组装切点规则
        StringBuilder expressionBuilder = null;
        if (StringUtils.hasText(properties.getPointcutPackages())) {
            if (properties.getPointcutPackages().contains(PACKAGE_DELIMITER)) {
                String[] nativeRpcPackageSplit = properties.getPointcutPackages().split(PACKAGE_DELIMITER);
                for (String nativeRpcPackageString : nativeRpcPackageSplit) {
                    if (StringUtils.hasText(nativeRpcPackageString)) {
                        if (null == expressionBuilder) {
                            expressionBuilder = new StringBuilder();
                            expressionBuilder.append("(execution(* ").append(nativeRpcPackageString).append("..*(..)))");
                        } else {
                            expressionBuilder.append(" || (execution(* ").append(nativeRpcPackageString).append("..*(..)))");
                        }
                    }
                }
            } else {
                expressionBuilder = new StringBuilder();
                expressionBuilder.append("(execution(* ").append(properties.getPointcutPackages()).append("..*(..)))");
            }
        }
        String expression = String.valueOf(expressionBuilder);
        advisor.setExpression(expression);
        advisor.setAdvice(new TanggulaLogAdvice(tanggulaClient, properties));
        advisor.setOrder(Integer.MAX_VALUE - 200);
        return advisor;
    }


}
