package com.tydic.dyc.pro.base.core.apollo;

import javax.crypto.Cipher;
import javax.crypto.spec.GCMParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.nio.charset.StandardCharsets;
import java.util.Base64;
import java.security.SecureRandom;

/**
 * AES-256-GCM decryptor with URL-safe Base64 (no padding) payloads.
 * Payload format: Base64Url( IV(12B) || Ciphertext || Tag(16B) )
 */
public class AesGcmCrypto {

    private static final String TRANSFORMATION = "AES/GCM/NoPadding";
    private static final int GCM_TAG_LENGTH_BITS = 128; // 16 bytes
    private static final int IV_LENGTH = 12;
    private static final int TAG_LENGTH = 16;
    private static final SecureRandom RANDOM = new SecureRandom();

    public static String decryptBase64Url(String base64UrlPayload, byte[] keyBytes) throws Exception {
        byte[] payload = Base64.getUrlDecoder().decode(normalizeBase64Url(base64UrlPayload));
        if (payload.length < IV_LENGTH + TAG_LENGTH) {
            throw new IllegalArgumentException("Invalid GCM payload length");
        }
        byte[] iv = new byte[IV_LENGTH];
        byte[] tag = new byte[TAG_LENGTH];
        byte[] ciphertext = new byte[payload.length - IV_LENGTH - TAG_LENGTH];

        System.arraycopy(payload, 0, iv, 0, IV_LENGTH);
        System.arraycopy(payload, IV_LENGTH, ciphertext, 0, ciphertext.length);
        System.arraycopy(payload, IV_LENGTH + ciphertext.length, tag, 0, TAG_LENGTH);

        // Java GCM expects tag appended to ciphertext implicitly; many libs append tag to end of ciphertext.
        // Here, we provide GCMParameterSpec and feed ciphertext only.
        GCMParameterSpec spec = new GCMParameterSpec(GCM_TAG_LENGTH_BITS, iv);
        SecretKeySpec keySpec = new SecretKeySpec(keyBytes, "AES");
        Cipher cipher = Cipher.getInstance(TRANSFORMATION);
        cipher.init(Cipher.DECRYPT_MODE, keySpec, spec);
        // Java cipher expects tag concatenated or supplied via spec; with JCE it's implicit from spec, ciphertext should include tag at end only when using updateAAD; here not needed.
        // Some implementations expect ciphertext||tag as input; if decryption fails, try the concat variant.
        try {
            return new String(cipher.doFinal(concat(ciphertext, tag)), StandardCharsets.UTF_8);
        } catch (Exception e) {
            // fallback attempt with ciphertext only (if provider expects separate tag)
            Cipher cipher2 = Cipher.getInstance(TRANSFORMATION);
            cipher2.init(Cipher.DECRYPT_MODE, keySpec, spec);
            return new String(cipher2.doFinal(ciphertext), StandardCharsets.UTF_8);
        }
    }

    public static String encryptBase64Url(String plaintext, byte[] keyBytes) throws Exception {
        byte[] iv = new byte[IV_LENGTH];
        RANDOM.nextBytes(iv);
        SecretKeySpec keySpec = new SecretKeySpec(keyBytes, "AES");
        GCMParameterSpec spec = new GCMParameterSpec(GCM_TAG_LENGTH_BITS, iv);
        Cipher cipher = Cipher.getInstance(TRANSFORMATION);
        cipher.init(Cipher.ENCRYPT_MODE, keySpec, spec);
        byte[] ctAndTag = cipher.doFinal(plaintext.getBytes(StandardCharsets.UTF_8));
        // Split ciphertext and tag: last 16 bytes are tag
        if (ctAndTag.length < TAG_LENGTH) {
            throw new IllegalStateException("GCM output too short");
        }
        int ctLen = ctAndTag.length - TAG_LENGTH;
        byte[] ciphertext = new byte[ctLen];
        byte[] tag = new byte[TAG_LENGTH];
        System.arraycopy(ctAndTag, 0, ciphertext, 0, ctLen);
        System.arraycopy(ctAndTag, ctLen, tag, 0, TAG_LENGTH);
        byte[] payload = concat(iv, concat(ciphertext, tag));
        return toBase64UrlNoPadding(payload);
    }

    private static String normalizeBase64Url(String s) {
        // Accept no-padding input. Base64 decoder in Java handles missing padding for URL variant in most cases,
        // but we can add padding if needed.
        int mod = s.length() % 4;
        if (mod == 2) return s + "==";
        if (mod == 3) return s + "=";
        return s;
    }

    private static byte[] concat(byte[] a, byte[] b) {
        byte[] out = new byte[a.length + b.length];
        System.arraycopy(a, 0, out, 0, a.length);
        System.arraycopy(b, 0, out, a.length, b.length);
        return out;
    }

    private static String toBase64UrlNoPadding(byte[] bytes) {
        return Base64.getUrlEncoder().withoutPadding().encodeToString(bytes);
    }
}
