package com.tydic.dyc.pro.base.core.apollo;

import java.nio.charset.StandardCharsets;
import java.security.SecureRandom;
import java.util.Base64;

/**
 * 简易命令行工具：生成密钥、加密、解密。
 *
 * 用法示例（需先提供运行时密钥，见 SecretKeyProvider）：
 * 1) 生成随机32字节密钥：
 *    main: gen-key
 * 2) 加密明文：
 *    main: encrypt your_plain_text
 *    输出：ENC(<Base64Url payload>) 可直接放到 Apollo
 * 3) 解密：
 *    main: decrypt ENC(<payload>)
 *    或   decrypt <payload>
 */
public class CryptoTool {
    private static final SecureRandom RANDOM = new SecureRandom();

    public static void main(String[] args) throws Exception {
        if (args == null || args.length == 0) {
            printHelp();
            return;
        }

        String cmd = args[0].toLowerCase();
        switch (cmd) {
            case "gen-key":
                genKey();
                break;
            case "encrypt":
                if (args.length < 2) {
                    System.out.println("缺少明文参数，如: encrypt myPassword");
                    return;
                }
                encrypt(joinFrom(args, 1));
                break;
            case "decrypt":
                if (args.length < 2) {
                    System.out.println("缺少密文参数，如: decrypt ENC(....)");
                    return;
                }
                decrypt(joinFrom(args, 1));
                break;
            default:
                printHelp();
        }
    }

    private static void genKey() {
        byte[] key = new byte[32];
        RANDOM.nextBytes(key);
        String b64u = Base64.getUrlEncoder().withoutPadding().encodeToString(key);
        System.out.println("生成32字节密钥：");
        System.out.println("Base64Url(推荐，设置为 APP_SECRET_KEY)：" + b64u);
    }

    private static void encrypt(String plaintext) throws Exception {
        byte[] key = SecretKeyProvider.loadKey();
        String payload = AesGcmCrypto.encryptBase64Url(plaintext, key);
        System.out.println("ENC(" + payload + ")");
    }

    private static void decrypt(String input) throws Exception {
        String payload = extractPayload(input.trim());
        byte[] key = SecretKeyProvider.loadKey();
        String plain = AesGcmCrypto.decryptBase64Url(payload, key);
        System.out.println(plain);
    }

    private static String extractPayload(String s) {
        if (s.startsWith("ENC(") && s.endsWith(")")) {
            return s.substring(4, s.length() - 1);
        }
        return s;
    }

    private static void printHelp() {
        System.out.println("用法:\n  gen-key\n  encrypt <plaintext>\n  decrypt <ENC(payload)|payload>\n密钥来源：环境变量 APP_SECRET_KEY 或文件 APP_SECRET_KEY_FILE (默认 /etc/app/secret.key)");
    }

    private static String joinFrom(String[] arr, int startIdx) {
        StringBuilder sb = new StringBuilder();
        for (int i = startIdx; i < arr.length; i++) {
            if (i > startIdx) sb.append(' ');
            sb.append(arr[i]);
        }
        return sb.toString();
    }
}
