package com.tydic.dyc.pro.base.core.apollo;

import org.springframework.boot.SpringApplication;
import org.springframework.boot.env.EnvironmentPostProcessor;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MutablePropertySources;
import org.springframework.core.env.PropertySource;

public class EncryptedPropertyEnvironmentPostProcessor implements EnvironmentPostProcessor {
    @Override
    public void postProcessEnvironment(ConfigurableEnvironment environment, SpringApplication application) {
        MutablePropertySources sources = environment.getPropertySources();

        // Install a top-level decrypting property source to handle ENC(...) for any sources added later (e.g. Apollo)
        String topName = "encDecTop";
        if (sources.get(topName) == null) {
            sources.addFirst(new TopDecryptingPropertySource(topName, sources));
            System.out.println("[ENC-DEC] addFirst: " + topName + " -> TopDecryptingPropertySource");
        }

        // Keep wrapping existing sources to maximize coverage for early-registered sources
        System.out.println("[ENC-DEC] init wrapping property sources ...");
        int wrappedCount = 0;
        int skippedCount = 0;
        for (PropertySource<?> ps : sources) {
            // Avoid wrapping Spring Boot's special property source used by Binder
            if ("org.springframework.boot.context.properties.source.ConfigurationPropertySourcesPropertySource"
                    .equals(ps.getClass().getName())) {
                System.out.println("[ENC-DEC] skip: " + ps.getName() + " {" + ps.getClass().getName() + "}");
                skippedCount++;
                continue;
            }
            // Also skip our top-level proxy itself
            if (topName.equals(ps.getName())) {
                skippedCount++;
                continue;
            }
            PropertySource<?> wrapped = DecryptingPropertySource.wrap(ps);
            if (wrapped != ps) {
                sources.replace(ps.getName(), wrapped);
                System.out.println("[ENC-DEC] wrap: " + ps.getName() + " -> " + wrapped.getClass().getSimpleName());
                wrappedCount++;
            } else {
                System.out.println("[ENC-DEC] passthrough: " + ps.getName() + " {" + ps.getClass().getSimpleName() + "}");
                skippedCount++;
            }
        }
        System.out.println("[ENC-DEC] done. wrapped=" + wrappedCount + ", skipped=" + skippedCount);
    }

    /**
     * A top-level PropertySource that delegates reads to underlying sources and decrypts any ENC(...) string values.
     * This avoids timing issues for sources (e.g. Apollo) that join after EnvironmentPostProcessor execution.
     */
    static class TopDecryptingPropertySource extends PropertySource<MutablePropertySources> {
        private static final String PREFIX = "ENC(";
        private static final String SUFFIX = ")";
        // Prevent re-entrant resolution loops when underlying sources delegate back into Environment
        private static final ThreadLocal<Boolean> IN_RESOLUTION = ThreadLocal.withInitial(() -> Boolean.FALSE);

        TopDecryptingPropertySource(String name, MutablePropertySources sources) {
            super(name, sources);
        }

        @Override
        public Object getProperty(String name) {
            // Reentrancy guard: if we're already resolving on this thread, don't participate
            if (Boolean.TRUE.equals(IN_RESOLUTION.get())) {
                return null;
            }
            IN_RESOLUTION.set(Boolean.TRUE);
            try {
                // Iterate through underlying sources (excluding this one) and return the first hit
                for (PropertySource<?> ps : this.source) {
                    if (ps == this) {
                        continue;
                    }
                    // Broadly skip Spring Boot configuration property sources to avoid delegation cycles
                    String cn = ps.getClass().getName();
                    if (cn != null && cn.startsWith("org.springframework.boot.context.properties.source")) {
                        continue;
                    }
                    Object v = ps.getProperty(name);
                    if (v == null) {
                        continue;
                    }
                    if (!(v instanceof String)) {
                        return v;
                    }
                    String s = ((String) v).trim();
                    if (!s.startsWith(PREFIX) || !s.endsWith(SUFFIX)) {
                        return v;
                    }
                    System.out.println("[ENC-DEC] hit ENC for key='" + name + "' in PS='" + ps.getName() + "'");
                    String cached = DecryptingCache.get(name, s);
                    if (cached != null) {
                        System.out.println("[ENC-DEC] cache hit for key='" + name + "'");
                        return cached;
                    }
                    String payload = s.substring(PREFIX.length(), s.length() - SUFFIX.length());
                    try {
                        byte[] key = SecretKeyProvider.loadKey();
                        String plain = AesGcmCrypto.decryptBase64Url(payload, key);
                        DecryptingCache.put(name, s, plain);
                        System.out.println("[ENC-DEC] decrypt ok for key='" + name + "'");
                        return plain;
                    } catch (Exception e) {
                        System.out.println("[ENC-DEC] decrypt fail for key='" + name + "' err=" + e.getMessage());
                        return v; // transparent fallback
                    }
                }
                return null;
            } finally {
                IN_RESOLUTION.set(Boolean.FALSE);
            }
        }
    }
}
