package com.tydic.dyc.oc.components.es;


import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.tydic.dyc.base.constants.BaseRspConstant;
import com.tydic.dyc.base.exception.BaseBusinessException;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpEntity;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.util.EntityUtils;
import org.elasticsearch.action.admin.indices.create.CreateIndexRequest;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.*;
import org.elasticsearch.xcontent.XContentType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.io.IOException;


/**
 * 标题：es获取客户端工具类
 * <p>
 * 说明：es获取客户端工具类
 * <br>
 * 时间：2019/1/1616:52<br>
 *
 *
 * </p>
 *
 * @author hegy
 */
@Component
@Slf4j
public class UocElasticsearchUtil {

    @Autowired
    private RestHighLevelClient restHighLevelClient;


    /**
     * 删除索引
     *
     * @param index 索引
     * @return boolean
     */
    public boolean deleteIndex(String index) {
        if (StringUtils.isBlank(index)) {
            log.info("删除索引内容为空");
            return false;
        }
        DeleteIndexRequest request = new DeleteIndexRequest(index);
        try {
//            DeleteIndexResponse indexResponse = restHighLevelClient.indices().delete(request, RequestOptions.DEFAULT); // 6.x jar包
            AcknowledgedResponse indexResponse = restHighLevelClient.indices().delete(request, RequestOptions.DEFAULT);// 升级7.10 jar包
            if (indexResponse.isAcknowledged()) {
                log.info("删除索引成功");
            } else {
                log.info("删除索引失败");
            }
            return indexResponse.isAcknowledged();
        } catch (Exception e) {
            log.error("删除索引失败", e);
            String exceptionStr = JSON.toJSONString(e.getMessage());
            if (exceptionStr.contains("index_not_found_exception")) {
                return true;
            }
        }
        return false;
    }


    /**
     * 创建索引
     *
     * @param index 索引
     * @return boolean
     */
    public boolean createIndex(String index) {
        //index名必须全小写，否则报错
        CreateIndexRequest request = new CreateIndexRequest(index);
        try {
            CreateIndexResponse indexResponse = restHighLevelClient.indices().create(request, RequestOptions.DEFAULT);
            if (indexResponse.isAcknowledged()) {
                log.info("创建索引成功");
            } else {
                log.info("创建索引失败");
            }
            return indexResponse.isAcknowledged();
        } catch (IOException e) {
            log.error("创建索引失败", e);
        }
        return false;
    }

    /**
     * 插入数据
     *
     * @param index  索引
     * @param type   类型
     * @param object 数据
     * @return String
     */
    public String addData(String index, String type, String id, JSONObject object) {
//        IndexRequest indexRequest = new IndexRequest(index, type, id);// 6.x jar包
        IndexRequest indexRequest = new IndexRequest(index);
        indexRequest.id(id);
        try {
//            indexRequest.source(mapper.writeValueAsString(object), XContentType.JSON);
            indexRequest.source(JSON.toJSONString(object), XContentType.JSON);
            IndexResponse indexResponse = restHighLevelClient.index(indexRequest, RequestOptions.DEFAULT);
            return indexResponse.getId();
        } catch (Exception e) {
            log.info("插入数据失败", e);
        }
        return null;
    }

    /**
     * 描述:保存json数据
     *
     * @param index   索引名称
     * @param id      索引数据id
     * @param jsonStr json数据
     * @return java.lang.String
     * @author tgy
     * @date 2022/3/1 13:00
     */
    public String addJsonData(String index, String id, String jsonStr) {
        IndexRequest indexRequest = new IndexRequest(index);
        indexRequest.id(id);
        try {
            indexRequest.source(jsonStr, XContentType.JSON);
            
            log.info("addJsonData保存数据：" + jsonStr);
            log.info("es存储源数据：" + indexRequest.source().utf8ToString());
            IndexResponse indexResponse = restHighLevelClient.index(indexRequest, RequestOptions.DEFAULT);
            log.info("es存储返回结果：" + indexResponse.getResult());
            
            return indexResponse.getId();
        } catch (Exception e) {
            log.error("插入数据失败", e);
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "插入数据失败：" + e.getMessage(), e);
        }
    }


    /**
     * 检查索引
     *
     * @param index 索引
     * @return boolean
     */
    public boolean checkIndexExist(String index) {
        String ok = "OK";
        boolean checkResult = false;
        try {
            Request request = new Request("HEAD", index);
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
            if (ok.equals(response.getStatusLine().getReasonPhrase())) {
                checkResult = true;
            }
        } catch (IOException e) {
            log.error("检查索引失败", e);
        }
        return checkResult;
    }

    /**
     * 根据条件删除ES数据
     *
     * @param endPoint 接口index点
     * @param qryJson  删除数据语句
     * @return boolean
     */
    public Boolean deleteDataByCondition(String endPoint, JSONObject qryJson) {
        boolean deleteResult = false;
        HttpEntity entity = new NStringEntity(qryJson.toJSONString(), ContentType.APPLICATION_JSON);
        try {
            Request request = new Request("POST", endPoint);
            request.setEntity(entity);
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
            String result = EntityUtils.toString(response.getEntity());
            if (StringUtils.isNotBlank(result)) {
                JSONObject deleteResultJson = JSONObject.parseObject(result);
                int deleteCount = deleteResultJson.getInteger("deleted");
                if (deleteCount > 0) {
                    deleteResult = true;
                }
            }
        } catch (IOException e) {
            log.error("根据条件删除ES数据失败", e);
        }
        return deleteResult;
    }

    public Boolean updateDataById(String endPoint, JSONObject updateJson, String id) {
        boolean updateREsult = false;
        //组装更新语句
        String esUpdateStr = toEsUpdateStr(id, updateJson);
        log.info("esUpdate URL:{}", endPoint);
        log.info("esUpdateStrReq:{}", esUpdateStr);
        HttpEntity entity = new NStringEntity(esUpdateStr, ContentType.APPLICATION_JSON);
        try {
            Request request = new Request("POST", endPoint);
            request.setEntity(entity);
            Response response = restHighLevelClient.getLowLevelClient().performRequest(request);
            String result = EntityUtils.toString(response.getEntity());
            log.info("esUpdateStrRsp:{}", result);
            if (StringUtils.isNotBlank(result)) {
                JSONObject rspJsonObject = JSON.parseObject(result);
                if ("1".equals(rspJsonObject.get("updated"))) {
                    updateREsult = true;
                }
            }
        } catch (IOException e) {
            log.error("根据条件更新ES数据失败:{},更新语句:{}", e, esUpdateStr);
        }
        return updateREsult;
    }

    /**
     * 获取低水平客户端
     *
     * @return RestClient
     */
    public RestClient getLowLevelClient() {
        return restHighLevelClient.getLowLevelClient();

    }

    /**
     * 组装更新es语句
     *
     * @param id           主键ID
     * @param updateParams 更新的字段
     * @return 更新es语句
     */
    public String toEsUpdateStr(String id, JSONObject updateParams) {
        //组装查询条件
        JSONObject term = new JSONObject();
        term.put("_id", id);

        JSONObject query = new JSONObject();
        query.put("term", term);

        JSONObject script = new JSONObject();
        script.put("source", getSource(updateParams));
        script.put("lang", "painless");
        script.put("params", updateParams);

        JSONObject esJsonObj = new JSONObject();
        esJsonObj.put("query", query);
        esJsonObj.put("script", script);

        return esJsonObj.toJSONString();
    }


    /**
     * 组装source映射字段
     *
     * @param updateParams 更新的字段
     * @return source值
     */
    private String getSource(JSONObject updateParams) {
        StringBuilder stringBuilder = new StringBuilder();
        updateParams.keySet().forEach(key -> {
            stringBuilder.append("ctx._source.")
                    .append(key)
                    .append("=")
                    .append("params.")
                    .append(key)
                    .append(";");
        });
        return stringBuilder.toString();
    }
}
