package com.tydic.dyc.oc.service.domainservice;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.tydic.dyc.base.constants.BaseRspConstant;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.oc.constants.*;
import com.tydic.dyc.oc.model.order.IUocOrderModel;
import com.tydic.dyc.oc.model.order.qrybo.UocConfEffectiveQryBo;
import com.tydic.dyc.oc.model.order.sub.UocConfEffective;
import com.tydic.dyc.oc.model.order.sub.UocOrderAccessory;
import com.tydic.dyc.oc.model.saleorder.IUocSaleOrderModel;
import com.tydic.dyc.oc.model.saleorder.UocSaleOrderDo;
import com.tydic.dyc.oc.model.saleorder.qrybo.UocSaleOrderItemQryBo;
import com.tydic.dyc.oc.model.saleorder.qrybo.UocSaleOrderQryBo;
import com.tydic.dyc.oc.model.saleorder.sub.UocSaleOrderItem;
import com.tydic.dyc.oc.model.shiporder.IUocShipOrderModel;
import com.tydic.dyc.oc.model.shiporder.UocShipOrderDo;
import com.tydic.dyc.oc.model.shiporder.qrybo.UocShipOrderQryBo;
import com.tydic.dyc.oc.model.shiporder.sub.UocShipOrderItem;
import com.tydic.dyc.oc.service.domainservice.bo.UocSaleShipOrderReqBo;
import com.tydic.dyc.oc.service.domainservice.bo.UocSaleShipOrderReqItemBo;
import com.tydic.dyc.oc.service.domainservice.bo.UocSaleShipOrderRspBo;
import com.tydic.dyc.oc.utils.IdUtil;
import lombok.extern.ohaotian.HTServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 描述: 〈领域服务-发货（基于销售单）实现〉
 *
 * @author : yuhao
 * @date : 2022/8/23 17:00
 */
@HTServiceImpl
public class UocSaleShipOrderServiceImpl implements UocSaleShipOrderService {

    @Autowired
    private IUocShipOrderModel iUocShipOrderModel;
    @Autowired
    private IUocSaleOrderModel iUocSaleOrderModel;
    @Autowired
    private IUocOrderModel iUocOrderModel;
    /**
     * 已发完货
     */
    private static final String OP_FLAG_YES = "1";
    /**
     * 未发完货
     */
    private static final String OP_FLAG_NO = "0";
    /**
     * 订单来源-电商
     */
    private static final String EC_IMPORT = "2";
    /**
     * 全部-供应商ID
     */
    private static final String SUP_NO = "0";

    @Override
    public UocSaleShipOrderRspBo dealImplShipOrder(UocSaleShipOrderReqBo reqBo) {
        //参数校验
        this.validateArg(reqBo);
        //获取销售单信息
        UocSaleOrderDo saleOrderInfo = this.getSaleOrderInfo(reqBo);
        //获取销售单明细信息
        List<UocSaleOrderItem> saleOrderItemInfos = this.getSaleOrderItemInfos(reqBo);
        //生成发货单id
        Long shipOrderId = IdUtil.nextId();
        //生成发货单编号
        String shipOrderNo = this.getShipOrderNo(saleOrderInfo.getOrderSource());
        //创建发货单
        this.createShipOrder(reqBo, saleOrderItemInfos, shipOrderId, shipOrderNo);
        //更新销售单发货数量
        this.updateSaleOrderSendCount(reqBo);
        //判断销售单发货是否完成
        boolean finish = this.checkSaleOrderIsFinish(reqBo);
        if (finish && !UocStateConstants.SaleOrder.XS_FH_FHWC.equals(saleOrderInfo.getSaleOrderState())) {
            //如果发货完成，需更更新销售单状态
            this.updateSaleOrderState(reqBo, UocStateConstants.SaleOrder.XS_FH_FHWC);
        } else {
            //发货未完成，也要更新未发货中
            if (UocStateConstants.SaleOrder.XS_FH_DFH.equals(saleOrderInfo.getSaleOrderState())) {
                //如果发货完成，需更更新销售单状态
                this.updateSaleOrderState(reqBo, UocStateConstants.SaleOrder.XS_FH_FHZ);
            }
        }
        //返回结果
        UocSaleShipOrderRspBo rspBo = new UocSaleShipOrderRspBo();
        rspBo.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
        rspBo.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
        rspBo.setShipOrderId(shipOrderId);
        rspBo.setShipOrderNo(shipOrderNo);
        rspBo.setFinish(finish);
        if (finish) {
            rspBo.setOpFlag(OP_FLAG_YES);
        } else {
            rspBo.setOpFlag(OP_FLAG_NO);
        }
        return rspBo;
    }


    /**
     * 更新销售单状态
     *
     * @param reqBo
     * @param saleOrderState
     */
    private void updateSaleOrderState(UocSaleShipOrderReqBo reqBo, String saleOrderState) {
        UocSaleOrderDo updateSaleOrderDo = new UocSaleOrderDo();
        updateSaleOrderDo.setOrderId(reqBo.getOrderId());
        updateSaleOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
        updateSaleOrderDo.setUpdateOperId(reqBo.getUserId().toString());
        updateSaleOrderDo.setSaleOrderState(saleOrderState);
        this.iUocSaleOrderModel.updateSaleStatusAndProCode(updateSaleOrderDo);
    }

    /**
     * 判断销售单是否发货完成
     *
     * @param reqBo
     * @return
     */
    private boolean checkSaleOrderIsFinish(UocSaleShipOrderReqBo reqBo) {
        //查询销售明细各种数量汇总
        UocSaleOrderItemQryBo uocSaleOrderItemQryBo = new UocSaleOrderItemQryBo();
        uocSaleOrderItemQryBo.setOrderId(reqBo.getOrderId());
        uocSaleOrderItemQryBo.setSaleOrderId(reqBo.getSaleOrderId());
        UocSaleOrderDo uocSaleOrderDo = this.iUocSaleOrderModel.getCollectCountImplOrderItem(uocSaleOrderItemQryBo);
        if (null != uocSaleOrderDo && !CollectionUtils.isEmpty(uocSaleOrderDo.getSaleOrderItems()) && null != uocSaleOrderDo.getSaleOrderItems().get(0)) {
            //采购数量 = 发货数量 + 退货数量 则发货完成
            if (uocSaleOrderDo.getSaleOrderItems().get(0).getPurchaseCount().compareTo(uocSaleOrderDo.getSaleOrderItems().get(0).getSendCount().add(uocSaleOrderDo.getSaleOrderItems().get(0).getReturnCount())) == 0) {
                return true;
            } else {
                return false;
            }
        } else {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_CONFIRM_REFUSE_ERROR, "查询销售单明细数量汇总为空");
        }
    }

    /**
     * 更新销售单发货数量
     *
     * @param reqBo
     */
    private void updateSaleOrderSendCount(UocSaleShipOrderReqBo reqBo) {
        UocSaleOrderDo updateSaleOrderDo = new UocSaleOrderDo();
        updateSaleOrderDo.setOrderId(reqBo.getOrderId());
        updateSaleOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
        updateSaleOrderDo.setUpdateOperId(String.valueOf(reqBo.getUserId()));
        updateSaleOrderDo.setUpdateTime(new Date());
        updateSaleOrderDo.setSaleOrderItems(reqBo.getShipOrderItemBoList().stream().map(uocImplShipOrderReqItemBo -> {
            UocSaleOrderItem uocSaleOrderItem = new UocSaleOrderItem();
            uocSaleOrderItem.setSaleOrderItemId(uocImplShipOrderReqItemBo.getSaleOrderItemId());
            uocSaleOrderItem.setSendCount(uocImplShipOrderReqItemBo.getSendCount());
            return uocSaleOrderItem;
        }).collect(Collectors.toList()));

        this.iUocSaleOrderModel.modifyBatchItemSendCount(updateSaleOrderDo);
    }

    /**
     * 生成发货单
     *
     * @param reqBo
     * @param saleOrderItemInfos
     * @param shipOrderId
     * @param shipOrderNo
     */
    private void createShipOrder(UocSaleShipOrderReqBo reqBo, List<UocSaleOrderItem> saleOrderItemInfos, Long shipOrderId, String shipOrderNo) {
        //将销售单明细list转化为map
        Map<Long, UocSaleOrderItem> saleOrderItemMap = saleOrderItemInfos.stream().collect(Collectors.toMap(UocSaleOrderItem::getSaleOrderItemId, o -> o));
        //组装发货单信息
        UocShipOrderDo createShipOrderDo = JSONObject.parseObject(JSON.toJSONString(reqBo), UocShipOrderDo.class);
        //设置发货单状态
        createShipOrderDo.setShipOrderState(UocStateConstants.ShipOrder.FH_FH_FH);
        //设置发货人id
        createShipOrderDo.setShipOperId(String.valueOf(reqBo.getUserId()));
        //设置发货单创建人
        createShipOrderDo.setCreateOperId(String.valueOf(reqBo.getUserId()));
        //设置发货单创建时间
        createShipOrderDo.setCreateTime(new Date());
        //设置发货单id
        createShipOrderDo.setShipOrderId(shipOrderId);
        //设置供应商id
        createShipOrderDo.setSupNo(saleOrderItemInfos.get(0).getSupplierId());
        //设置发货单编号
        createShipOrderDo.setShipOrderNo(shipOrderNo);

        //设置发货单附件
        if (!CollectionUtils.isEmpty(reqBo.getOrderAccessoryBoList())) {
            List<UocOrderAccessory> uocOrderAccessoryList = JSONArray.parseArray(JSON.toJSONString(reqBo.getOrderAccessoryBoList())).toJavaList(UocOrderAccessory.class);
            uocOrderAccessoryList.forEach(uocOrderAccessory -> {
                uocOrderAccessory.setOrderId(reqBo.getOrderId());
                uocOrderAccessory.setCreateTime(new Date());
                uocOrderAccessory.setCreateOperId(reqBo.getUserId().toString());
                uocOrderAccessory.setId(IdUtil.nextId());
                uocOrderAccessory.setObjType(UocDicConstant.OBJ_TYPE.SHIP);
                uocOrderAccessory.setObjId(shipOrderId);
                uocOrderAccessory.setAttachmentType(UocAttachementTypeConstants.DELIVERY);
            });
            createShipOrderDo.setOrderAccessoryBoList(uocOrderAccessoryList);
        }

        //设置发货单明细
        createShipOrderDo.getShipOrderItemBoList().forEach(uocShipOrderItem -> {
            UocSaleOrderItem uocSaleOrderItem = saleOrderItemMap.get(uocShipOrderItem.getSaleOrderItemId());
            //校验是否存在已发货完的商品
            if (uocShipOrderItem.getSendCount().add(uocSaleOrderItem.getSendCount()).compareTo(uocSaleOrderItem.getPurchaseCount()) > 0) {
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_IMPL_SHIP_ORDER_ERROR, "存在已发货完成的商品，请重新选择商品发货");
            }
            //设置发货单id
            uocShipOrderItem.setShipOrderId(shipOrderId);
            //设置订单id
            uocShipOrderItem.setOrderId(reqBo.getOrderId());
            //设置销售单id
            uocShipOrderItem.setSaleOrderId(reqBo.getSaleOrderId());
            //设置销售单明细id
            uocShipOrderItem.setSaleOrderItemId(uocSaleOrderItem.getSaleOrderItemId());
            //设置计量单位
            uocShipOrderItem.setUnitName(uocSaleOrderItem.getUnitName());
            //设置创建人id
            uocShipOrderItem.setCreateOperId(String.valueOf(reqBo.getUserId()));
            //设置创建时间
            uocShipOrderItem.setCreateTime(new Date());
            //设置发货单明细id
            uocShipOrderItem.setShipOrderItemId(IdUtil.nextId());
        });
        //设置时效配置
        createShipOrderDo.setAllowChngTime(this.getShipAllowChngTime(reqBo));

        //调用计算发货单信息
        this.computeShipOrderInfo(createShipOrderDo);

        //生产发货单
        this.iUocShipOrderModel.createShipOrder(createShipOrderDo);

        //调用修改销售单计算信息
        this.updateSaleComputeInfo(createShipOrderDo);
    }

    /**
     * 计算发货单信息
     */
    private void computeShipOrderInfo(UocShipOrderDo uocShipOrderDo) {
        //定义查询销售单主体参数
        UocSaleOrderQryBo uocSaleOrderQryBo = new UocSaleOrderQryBo();
        //设置订单ID
        uocSaleOrderQryBo.setOrderId(uocShipOrderDo.getOrderId());
        //设置销售单ID
        uocSaleOrderQryBo.setSaleOrderId(uocShipOrderDo.getSaleOrderId());
        //调用查询销售单主体
        UocSaleOrderDo uocSaleOrderDo = iUocSaleOrderModel.qrySaleOrder(uocSaleOrderQryBo);

        //定义查询销售单明细参数
        UocSaleOrderItemQryBo uocSaleOrderItemQryBo = new UocSaleOrderItemQryBo();
        //设置订单ID
        uocSaleOrderItemQryBo.setOrderId(uocShipOrderDo.getOrderId());
        //设置销售单ID
        uocSaleOrderItemQryBo.setSaleOrderId(uocShipOrderDo.getSaleOrderId());
        //调用查询销售单明细
        List<UocSaleOrderItem> uocSaleOrderItemList = iUocSaleOrderModel.getSaleOrderItemList(uocSaleOrderItemQryBo);

        //定义发货单列表查询参数
        UocShipOrderQryBo uocShipOrderQryBo = new UocShipOrderQryBo();
        //设置订单ID
        uocShipOrderQryBo.setOrderId(uocShipOrderDo.getOrderId());
        //设置销售单ID
        uocShipOrderQryBo.setSaleOrderId(uocShipOrderDo.getSaleOrderId());
        //调用发货单列表查询
        List<UocShipOrderDo> uocShipOrderDoList = iUocShipOrderModel.getListShipOrder(uocShipOrderQryBo);

        //调用计算发货单运费
        this.computeShipOrderTransFee(uocShipOrderDo, uocSaleOrderDo, uocSaleOrderItemList, uocShipOrderDoList);

        //调用计算发货单金额
        this.computeShipOrderFee(uocShipOrderDo, uocSaleOrderItemList, uocShipOrderDoList);
    }

    /**
     * 计算发货单运费
     */
    private void computeShipOrderTransFee(UocShipOrderDo uocShipOrderDo, UocSaleOrderDo uocSaleOrderDo, List<UocSaleOrderItem> uocSaleOrderItemList, List<UocShipOrderDo> uocShipOrderDoList) {
        //获取销售单明细 所有采购数量
        BigDecimal allPurchaseCount = uocSaleOrderItemList.stream().map(UocSaleOrderItem::getPurchaseCount).reduce(BigDecimal.ZERO, BigDecimal::add);

        //获取发货单明细 所有发货数量
        BigDecimal allSendCount = uocShipOrderDo.getShipOrderItemBoList().stream().map(UocShipOrderItem::getSendCount).reduce(BigDecimal.ZERO, BigDecimal::add);


        //判断是否没有运费
        if (uocSaleOrderDo.getTotalTransFee().compareTo(BigDecimal.ZERO) == 0) {
            //设置发货单运费
            uocShipOrderDo.setSendTransFee(BigDecimal.ZERO);

            return;
        }

        //判断是否只有一次发货
        if (CollectionUtils.isEmpty(uocShipOrderDoList) && allPurchaseCount.compareTo(allSendCount) == 0) {
            //设置发货单运费
            uocShipOrderDo.setSendTransFee(uocSaleOrderDo.getTotalTransFee());

            return;
        }

        //校验是否是最后一个发货单
        boolean isLast = this.checkShipOrderIsLast(uocShipOrderDo.getOrderId(), uocShipOrderDo.getSaleOrderId(), allSendCount);

        //判断是否是最后一个发货单
        if (isLast) {
            //计算发货单运费 = 销售单运费 - 已占用运费
            BigDecimal shipSendTransFee = uocSaleOrderDo.getTotalTransFee().subtract(uocSaleOrderDo.getTotalOccupyTransFee()).setScale(2, RoundingMode.HALF_UP);

            //设置发货单运费
            uocShipOrderDo.setSendTransFee(shipSendTransFee);

            return ;
        }

        //获取销售单明细ID Vs 销售单明细 Map
        Map<Long, UocSaleOrderItem> itemIdVsSaleOrderItemMap = uocSaleOrderItemList.stream().collect(Collectors.toMap(UocSaleOrderItem::getSaleOrderItemId, po -> po, (k1, k2) -> k2));

        //获取销售单明细 所有商品总重量
        BigDecimal allTotalGoodsWeight = uocSaleOrderItemList.stream().map(UocSaleOrderItem::getTotalGoodsWeight).reduce(BigDecimal.ZERO, BigDecimal::add);

        //获取销售单商品 运费单价 = 总运费 / 所有商品总重量
        BigDecimal freightUnitPrice = uocSaleOrderDo.getTotalTransFee().divide(allTotalGoodsWeight, 2, RoundingMode.HALF_UP);

        //获取发货单明细 所有商品总重量
        BigDecimal allShipTotalGoodsWeight = new BigDecimal("0");
        //遍历发货单明细
        for (UocShipOrderItem uocShipOrderItem : uocShipOrderDo.getShipOrderItemBoList()) {
            //获取当前发货单明细对应的销售单明细
            UocSaleOrderItem uocSaleOrderItem = itemIdVsSaleOrderItemMap.get(uocShipOrderItem.getSaleOrderItemId());

            //计算当前发货单明细 商品总重量 = 商品重量 * 发货数量
            BigDecimal shipTotalGoodsWeight = uocSaleOrderItem.getGoodsWeight().multiply(uocShipOrderItem.getSendCount());

            //计算发货单明细 所有商品总重量
            allShipTotalGoodsWeight = allShipTotalGoodsWeight.add(shipTotalGoodsWeight);
        }

        //计算发货单运费 = 运费单价 * 发货单明细 所有商品总重量
        BigDecimal shipSendTransFee = freightUnitPrice.multiply(allShipTotalGoodsWeight).setScale(2, RoundingMode.HALF_UP);

        //设置发货单运费
        uocShipOrderDo.setSendTransFee(shipSendTransFee);
    }

    /**
     * 计算发货单金额
     */
    private void computeShipOrderFee(UocShipOrderDo uocShipOrderDo, List<UocSaleOrderItem> uocSaleOrderItemList, List<UocShipOrderDo> uocShipOrderDoList) {
        //获取销售单明细 所有采购数量
        BigDecimal allPurchaseCount = uocSaleOrderItemList.stream().map(UocSaleOrderItem::getPurchaseCount).reduce(BigDecimal.ZERO, BigDecimal::add);

        //获取发货单明细 所有发货数量
        BigDecimal allSendCount = uocShipOrderDo.getShipOrderItemBoList().stream().map(UocShipOrderItem::getSendCount).reduce(BigDecimal.ZERO, BigDecimal::add);

        //获取销售单明细ID Vs 销售单明细 Map
        Map<Long, UocSaleOrderItem> itemIdVsSaleOrderItemMap = uocSaleOrderItemList.stream().collect(Collectors.toMap(UocSaleOrderItem::getSaleOrderItemId, po -> po, (k1, k2) -> k2));

        //获取发货单明细集合
        List<UocShipOrderItem> uocShipOrderItemList = uocShipOrderDo.getShipOrderItemBoList();

        //校验是否是最后一个发货单
        boolean isLast = this.checkShipOrderIsLast(uocShipOrderDo.getOrderId(), uocShipOrderDo.getSaleOrderId(), allSendCount);

        //遍历发货单明细集合
        for (UocShipOrderItem uocShipOrderItem : uocShipOrderItemList) {
            //获取当前发货单明细对应的销售单明细
            UocSaleOrderItem uocSaleOrderItem = itemIdVsSaleOrderItemMap.get(uocShipOrderItem.getSaleOrderItemId());

            //判断是否只有一次发货
            if (CollectionUtils.isEmpty(uocShipOrderDoList) && allPurchaseCount.compareTo(allSendCount) == 0) {
                //设置发货明细销售金额
                uocShipOrderItem.setSendSaleFee(uocSaleOrderItem.getSaleFee());
                //设置发货明细采购金额
                uocShipOrderItem.setSendPurchaseFee(uocSaleOrderItem.getPurchaseFee());

                continue;
            }

            //判断是否是最后一个发货单
            if (isLast) {
                //计算发货明细销售金额 = 销售总金额 - 已发货明细销售金额
                BigDecimal sendSaleFee = uocSaleOrderItem.getSaleFee().subtract(ObjectUtil.isNotEmpty(uocSaleOrderItem.getTotalSendGoodsSaleFee()) ? uocSaleOrderItem.getTotalSendGoodsSaleFee() : BigDecimal.ZERO).setScale(2, RoundingMode.HALF_UP);
                //计算发货明细采购金额 = 采购总金额 - 已发货明细采购金额
                BigDecimal sendPurchaseFee = uocSaleOrderItem.getPurchaseFee().subtract(ObjectUtil.isNotEmpty(uocSaleOrderItem.getTotalSendGoodsPurchaseFee()) ? uocSaleOrderItem.getTotalSendGoodsPurchaseFee() : BigDecimal.ZERO).setScale(2, RoundingMode.HALF_UP);

                //设置发货明细销售金额
                uocShipOrderItem.setSendSaleFee(sendSaleFee);
                //设置发货明细采购金额
                uocShipOrderItem.setSendPurchaseFee(sendPurchaseFee);

                continue;
            }

            //计算发货明细销售金额 = 销售单价金额 * 发货数量
            BigDecimal sendSaleFee = uocSaleOrderItem.getSalePrice().multiply(uocShipOrderItem.getSendCount()).setScale(2, RoundingMode.HALF_UP);
            //计算发货明细采购金额 = 采购单价金额 * 发货数量
            BigDecimal sendPurchaseFee = uocSaleOrderItem.getPurchasePrice().multiply(uocShipOrderItem.getSendCount()).setScale(2, RoundingMode.HALF_UP);

            //设置发货明细销售金额
            uocShipOrderItem.setSendSaleFee(sendSaleFee);
            //设置发货明细采购金额
            uocShipOrderItem.setSendPurchaseFee(sendPurchaseFee);
        }

        //设置发货单明细
        uocShipOrderDo.setShipOrderItemBoList(uocShipOrderItemList);
    }

    /**
     * 修改销售单计算信息
     */
    private void updateSaleComputeInfo(UocShipOrderDo uocShipOrderDo) {
        //修改销售单已占用运费
        this.updateSaleOccupyTransFee(uocShipOrderDo);

        //修改销售单明细发货商品金额
        this.updateSaleItemSendGoodsFee(uocShipOrderDo);
    }

    /**
     * 修改销售单已占用运费
     */
    private void updateSaleOccupyTransFee(UocShipOrderDo uocShipOrderDo) {
        //定义修改销售单已占用运费参数
        UocSaleOrderDo updateOccupyTransFee = new UocSaleOrderDo();
        //设置销售单ID
        updateOccupyTransFee.setSaleOrderId(uocShipOrderDo.getSaleOrderId());
        //设置订单ID
        updateOccupyTransFee.setOrderId(uocShipOrderDo.getOrderId());
        //设置已占用运费
        updateOccupyTransFee.setTotalOccupyTransFee(uocShipOrderDo.getSendTransFee());
        //调用修改销售单占用运费
        this.iUocSaleOrderModel.modifyOccupyTransFee(updateOccupyTransFee);
    }

    /**
     * 修改销售单明细发货商品金额
     */
    private void updateSaleItemSendGoodsFee(UocShipOrderDo uocShipOrderDo) {
        //获取销售单明细ID Vs 发货单明细 Map
        Map<Long, List<UocShipOrderItem>> itemIdVsShipOrderItemMap = uocShipOrderDo.getShipOrderItemBoList().stream().collect(Collectors.groupingBy(UocShipOrderItem::getSaleOrderItemId));

        //定义销售单明细集合
        List<UocSaleOrderItem> uocSaleOrderItemList = new ArrayList<>(1);

        //遍历销售单明细ID Vs 发货单明细 Map
        for (Map.Entry<Long, List<UocShipOrderItem>> entry : itemIdVsShipOrderItemMap.entrySet()) {
            //定义销售单明细对象
            UocSaleOrderItem uocSaleOrderItem = new UocSaleOrderItem();

            //获取当前发货单明细 所有发货明细销售金额
            BigDecimal allSendSaleFee = entry.getValue().stream().map(UocShipOrderItem::getSendSaleFee).reduce(BigDecimal.ZERO, BigDecimal::add);
            //获取当前发货单明细 所有发货明细采购金额
            BigDecimal allSendPurchaseFee = entry.getValue().stream().map(UocShipOrderItem::getSendPurchaseFee).reduce(BigDecimal.ZERO, BigDecimal::add);

            //设置销售单明细ID
            uocSaleOrderItem.setSaleOrderItemId(entry.getKey());
            //设置已发货明细销售金额
            uocSaleOrderItem.setTotalSendGoodsSaleFee(allSendSaleFee);
            //设置已发货明细采购金额
            uocSaleOrderItem.setTotalSendGoodsPurchaseFee(allSendPurchaseFee);

            //添加到销售单明细集合
            uocSaleOrderItemList.add(uocSaleOrderItem);
        }

        //定义修改销售单明细发货商品金额参数
        UocSaleOrderDo updateItemSendGoodsFee = new UocSaleOrderDo();
        //设置销售单ID
        updateItemSendGoodsFee.setSaleOrderId(uocShipOrderDo.getSaleOrderId());
        //设置订单ID
        updateItemSendGoodsFee.setOrderId(uocShipOrderDo.getOrderId());
        //设置销售单明细集合
        updateItemSendGoodsFee.setSaleOrderItems(uocSaleOrderItemList);
        //调用修改销售单明细已发货明细金额
        this.iUocSaleOrderModel.modifyItemTotalSendGoodsFee(updateItemSendGoodsFee);
    }

    /**
     * 校验是否是最后一个发货单
     * @param orderId 订单ID
     * @param saleOrderId 销售单ID
     * @param allSendCount 所有发货数量
     * @return 是否最后一次发货
     */
    private boolean checkShipOrderIsLast(Long orderId, Long saleOrderId, BigDecimal allSendCount) {
        //定义查询明细各种数量汇总参数
        UocSaleOrderItemQryBo uocSaleOrderItemQryBo = new UocSaleOrderItemQryBo();
        //设置销售单ID
        uocSaleOrderItemQryBo.setSaleOrderId(saleOrderId);
        //设置订单ID
        uocSaleOrderItemQryBo.setOrderId(orderId);
        //调用查询明细各种数量汇总
        UocSaleOrderDo uocSaleOrderDo = iUocSaleOrderModel.getCollectCountImplOrderItem(uocSaleOrderItemQryBo);

        if (org.apache.commons.collections.CollectionUtils.isEmpty(uocSaleOrderDo.getSaleOrderItems())) {
            throw new BaseBusinessException(BaseRspConstant.RSP_CODE_FAIL, "查询数量汇总为空");
        }

        //获取当前销售单明细数量汇总
        UocSaleOrderItem uocSaleOrderItem = uocSaleOrderDo.getSaleOrderItems().get(0);

        return uocSaleOrderItem.getPurchaseCount().compareTo(uocSaleOrderItem.getSendCount().add(allSendCount)) == 0;
    }

    /**
     * 获取发货单时效配置
     *
     * @param reqBo
     * @return
     */
    private Date getShipAllowChngTime(UocSaleShipOrderReqBo reqBo) {
        //查询销售单详情
        UocSaleOrderQryBo querySaleOrderQryBo = new UocSaleOrderQryBo();
        querySaleOrderQryBo.setSaleOrderId(reqBo.getSaleOrderId());
        querySaleOrderQryBo.setOrderId(reqBo.getOrderId());
        UocSaleOrderDo saleOrderInfo = this.iUocSaleOrderModel.qrySaleOrder(querySaleOrderQryBo);

        //查询时效类配置
        UocConfEffectiveQryBo uocConfEffectiveQryBo = new UocConfEffectiveQryBo();
        uocConfEffectiveQryBo.setEffectiveCode(UocConstant.EffectiveCode.TIME_LIMIT_FOR_ABNORMAL_ORDER_CHANGES);
        List<UocConfEffective> rows = this.iUocOrderModel.qryListPageConfEffective(uocConfEffectiveQryBo).getRows();

        Date allowChngTime = null;
        if (ObjectUtil.isNotEmpty(rows)) {
            //将时效配置转化为map
            Map<String, UocConfEffective> confEffectiveMap = rows.stream().collect(Collectors.toMap(o -> o.getBusinessTypeCode() + o.getSupplierNo(), Function.identity()));
            //找到对应配置的时效配置
            if (null != saleOrderInfo) {
                String key = null;
                if (null != saleOrderInfo.getStakeholder()) {
                    key = saleOrderInfo.getOrderSource() + saleOrderInfo.getStakeholder().getSupId();
                } else {
                    key = String.valueOf(saleOrderInfo.getOrderSource()) + reqBo.getSupId();
                }
                if (ObjectUtil.isNotNull(confEffectiveMap.get(key))) {
                    allowChngTime = new Date(confEffectiveMap.get(key).getAging());
                } else {
                    allowChngTime = new Date(confEffectiveMap.get(saleOrderInfo.getOrderSource() + SUP_NO).getAging());
                }
            } else {
                allowChngTime = new Date(confEffectiveMap.get(EC_IMPORT + SUP_NO).getAging());
            }
        }
        return allowChngTime;
    }


    /**
     * 查询销售单明细
     */
    private List<UocSaleOrderItem> getSaleOrderItemInfos(UocSaleShipOrderReqBo reqBo) {
        List<Long> saleOrderItemIds = reqBo.getShipOrderItemBoList().stream().map(UocSaleShipOrderReqItemBo::getSaleOrderItemId).collect(Collectors.toList());
        UocSaleOrderItemQryBo uocSaleOrderItemQryBo = new UocSaleOrderItemQryBo();
        uocSaleOrderItemQryBo.setSaleOrderItemIdList(saleOrderItemIds);
        uocSaleOrderItemQryBo.setOrderId(reqBo.getOrderId());
        uocSaleOrderItemQryBo.setSaleOrderId(reqBo.getSaleOrderId());
        UocSaleOrderDo uocSaleOrderDo = this.iUocSaleOrderModel.getListSaleOrderItem(uocSaleOrderItemQryBo);
        List<UocSaleOrderItem> saleOrderItems = uocSaleOrderDo.getSaleOrderItems();
        if (CollectionUtils.isEmpty(saleOrderItems)) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_IMPL_SHIP_ORDER_ERROR, "获取销售单明细为空");
        }
        return saleOrderItems;
    }

    /**
     * 获取发货单编号
     *
     * @param orderSource
     * @return
     */
    private String getShipOrderNo(Integer orderSource) {
        UocShipOrderDo uocShipOrderDo = new UocShipOrderDo();
        uocShipOrderDo.setOrderSource(orderSource);
        uocShipOrderDo = this.iUocShipOrderModel.getShipOrderNo(uocShipOrderDo);
        if (null != uocShipOrderDo) {
            return uocShipOrderDo.getShipOrderNo();
        } else {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_IMPL_SHIP_ORDER_ERROR, "生成发货单编号失败");
        }
    }

    /**
     * 获取
     *
     * @param reqBo
     * @return
     */
    private UocSaleOrderDo getSaleOrderInfo(UocSaleShipOrderReqBo reqBo) {
        UocSaleOrderDo uocSaleOrderDo = new UocSaleOrderDo();
        uocSaleOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
        uocSaleOrderDo.setOrderId(reqBo.getOrderId());
        uocSaleOrderDo = this.iUocSaleOrderModel.getSaleOrderMain(uocSaleOrderDo);
        if (null == uocSaleOrderDo) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_IMPL_SHIP_ORDER_ERROR, "查询销售单为空");
        }
        if (!UocStateConstants.SaleOrder.XS_FH_DFH.equals(uocSaleOrderDo.getSaleOrderState()) && !UocStateConstants.SaleOrder.XS_FH_FHZ.equals(uocSaleOrderDo.getSaleOrderState())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_IMPL_SHIP_ORDER_ERROR, "销售单状态不为待发货或发货中，不能发货");
        }
        return uocSaleOrderDo;
    }

    /**
     * 参数校验
     *
     * @param reqBo
     */
    private void validateArg(UocSaleShipOrderReqBo reqBo) {
        if (null == reqBo) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象为空");
        }
        if (null == reqBo.getOrderId()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参订单ID为空");
        }
        if (null == reqBo.getSaleOrderId()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参销售单ID为空");
        }
        if (null == reqBo.getUserId()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参用户ID为空");
        }

    }
}

