package com.tydic.dyc.oc.components.statecalculator;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.oc.components.statecalculator.bo.UocStateCalculatorReqBo;
import com.tydic.dyc.oc.components.statecalculator.bo.UocStateCalculatorRspBo;
import com.tydic.dyc.oc.components.statecalculator.bo.UocStateCalculatorRspDataBo;
import com.tydic.dyc.oc.constants.UocDicConstant;
import com.tydic.dyc.oc.constants.UocRspConstants;
import com.tydic.dyc.oc.model.common.IUocCommonModel;
import com.tydic.dyc.oc.model.common.qrybo.UocQryOrderStateChngLogBo;
import com.tydic.dyc.oc.model.common.qrybo.UocQrySaleOrderStageBO;
import com.tydic.dyc.oc.model.order.sub.UocOrderStateChgLog;
import com.tydic.dyc.oc.repository.UocCommonRepository;
import com.tydic.dyc.oc.utils.IdUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;

/**
 * 标题:UocCalculatorService
 * 说明:状态计算服务
 * 时间:2023/7/12 17:11
 * 作者:田桂银
 */
@Component
@Slf4j
public class UocStateCalculatorService {
    @Autowired
    private UocStateCalculatorManager uocStateCalculatorManager;
    @Autowired
    private UocCommonRepository uocCommonRepository;

    @Autowired
    private IUocCommonModel iUocCommonModel;

    //计算并保存状态
    public void calculateStateAndSaveLog(UocStateCalculatorReqBo reqBo) {
        log.info("状态计算服务入参：{}", JSON.toJSONString(reqBo));
        //参数校验
        validateReqArg(reqBo);
        String callServerClassName = reqBo.getCallServer().getClass().getName();
        //根据调用服务类路径，获取计算器服务映射配置（计算器类路径）
        String calculatorClassName = getCalculatorClassName(callServerClassName);
        if (ObjectUtil.isNotEmpty(calculatorClassName)) {
            //获取计算器实例
            UocStateCalculator uocStateCalculator = uocStateCalculatorManager.getUocStateCalculator(calculatorClassName);
            if (ObjectUtil.isNotEmpty(uocStateCalculator)) {
                log.info("状态计算服务实现类调用入参：{}", JSON.toJSONString(reqBo));
                //调用不同计算器进行状态计算
                UocStateCalculatorRspBo uocStateCalculatorRspBo = uocStateCalculator.calculateState(reqBo);
                log.info("状态计算服务实现类调用出参：{}", JSON.toJSONString(uocStateCalculatorRspBo));
                //出参校验
                validateRsp(uocStateCalculatorRspBo);
                if (ObjectUtil.isNotEmpty(uocStateCalculatorRspBo.getChangeDataList())) {
                    //记录状态变更日志
                    saveStateChngLog(uocStateCalculatorRspBo);
                }
            }
        }

    }

    /**
     * 记录状态变更日志
     */
    private void saveStateChngLog(UocStateCalculatorRspBo uocStateCalculatorRspBo) {
        List<UocStateCalculatorRspDataBo> changeDataList = uocStateCalculatorRspBo.getChangeDataList();
        for (UocStateCalculatorRspDataBo uocStateCalculatorRspDataBo : changeDataList) {
            UocOrderStateChgLog uocOrderStateChgLog = new UocOrderStateChgLog();
            uocOrderStateChgLog.setId(IdUtil.nextId());
            uocOrderStateChgLog.setObjId(uocStateCalculatorRspDataBo.getObjId());
            uocOrderStateChgLog.setOrderId(uocStateCalculatorRspDataBo.getOrderId());
            uocOrderStateChgLog.setObjType(uocStateCalculatorRspDataBo.getObjType());
            //查询旧状态
            uocOrderStateChgLog.setOldState(qryOldState(uocStateCalculatorRspDataBo));
            uocOrderStateChgLog.setNewState(uocStateCalculatorRspDataBo.getState());
            uocOrderStateChgLog.setChgTime(new Date());
            uocOrderStateChgLog.setOperId(uocStateCalculatorRspDataBo.getOperId());
            uocOrderStateChgLog.setChgReson(uocStateCalculatorRspDataBo.getChngReason());
            uocOrderStateChgLog.setChgDesc(uocStateCalculatorRspDataBo.getChngDesc());

            //调用通用repository，创建状态变更日志
            uocCommonRepository.createOrderStateChangeLog(uocOrderStateChgLog);
        }

    }

    /**
     * 查询旧状态
     */
    private String qryOldState(UocStateCalculatorRspDataBo uocStateCalculatorRspDataBo) {
        UocQryOrderStateChngLogBo qryBo = new UocQryOrderStateChngLogBo();
        qryBo.setObjId(uocStateCalculatorRspDataBo.getObjId());
        qryBo.setOrderId(uocStateCalculatorRspDataBo.getOrderId());
        qryBo.setObjType(uocStateCalculatorRspDataBo.getObjType());
        return iUocCommonModel.qryUocLastChngStateByObjId(qryBo).getNewState();
    }

    /**
     * 从数据库获取调用类，对应的计算器类路径
     */
    private String getCalculatorClassName(String callServerClassName) {
        return iUocCommonModel.qryUocStateCalculatorServerConfig(callServerClassName).getCalculatorService();
    }

    /**
     * 计算出参校验
     */
    private void validateRsp(UocStateCalculatorRspBo reqChngBo) {
        if (null == reqChngBo) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "订单状态计算出参校验失败：入参对象不能为空");
        }
        List<UocStateCalculatorRspDataBo> changeDataList = reqChngBo.getChangeDataList();
        if (ObjectUtil.isNotEmpty(changeDataList)) {
            for (UocStateCalculatorRspDataBo reqBo : changeDataList) {
                if (null == reqBo.getOrderId()) {
                    throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "订单状态计算出参校验失败：字段[orderId]不能为空");
                }
                if (null == reqBo.getObjType()) {
                    throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "订单状态计算出参校验失败：字段[objType]不能为空");
                }
                if (null == reqBo.getObjId()) {
                    throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "订单状态计算出参校验失败：字段[objId]不能为空");
                }
                if (null == reqBo.getState()) {
                    throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "订单状态计算出参校验失败：字段[state]不能为空");
                }
            }
        }

    }

    /**
     * 计算入参校验
     */
    private void validateReqArg(UocStateCalculatorReqBo reqBo) {
        if (null == reqBo) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "订单状态计算入参校验失败：入参对象不能为空");
        }
        if (null == reqBo.getCallServer()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "订单状态计算入参校验失败：字段[callServer]不能为空");
        }
        if (null == reqBo.getServerReqBo()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "订单状态计算入参校验失败：字段[serverReqBo]不能为空");
        }
        if (null == reqBo.getServerRspBo()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "订单状态计算入参校验失败：字段[serverRspBo]不能为空");
        }
    }
}


