package com.yomahub.liteflow.script;

import cn.hutool.core.util.ObjectUtil;
import com.yomahub.liteflow.enums.LanguageTypeEnum;
import com.yomahub.liteflow.script.exception.ScriptSpiException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.script.ScriptException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;

/**
 * 脚本执行器工厂类
 * @author Bryan.Zhang
 * @since 2.6.0
 */
public class ScriptExecutorFactory {

    private Logger log = LoggerFactory.getLogger(this.getClass());

    private static ScriptExecutorFactory scriptExecutorFactory;

    private ScriptExecutor scriptExecutor;

    private Map<String,ScriptExecutor> scriptExecutorMap = new HashMap<>();

    public static ScriptExecutorFactory loadInstance(){
        if (ObjectUtil.isNull(scriptExecutorFactory)){
            scriptExecutorFactory = new ScriptExecutorFactory();
        }
        return scriptExecutorFactory;
    }

    /**
     * 根据language获取
     * @param language
     * @return
     */
    public ScriptExecutor getScriptExecutor(LanguageTypeEnum language){
        if(language == null){
            //获取默认
            return getScriptExecutor();
        }
        ScriptExecutor scriptExecutor = scriptExecutorMap.get(language.getCode());
        if(ObjectUtil.isNull(scriptExecutor)){
            for (LanguageTypeEnum value : LanguageTypeEnum.values()) {
                try {
                    Class clzz = Class.forName(value.getClassName());
                    ScriptExecutor executorImpl = (ScriptExecutor) clzz.newInstance();
                    executorImpl.init();
                    log.info("加载脚本解析器:{}|{}",value.getCode(),value.getClassName());
                    scriptExecutorMap.put(value.getCode(),executorImpl);
                } catch (Exception e) {
                    log.warn("加载脚本解析器-异常:{}",e.getMessage());
                }
            }
        }
        scriptExecutor = scriptExecutorMap.get(language.getCode());
        if(scriptExecutor == null){
            log.error("加载脚本解析器-异常:{}|{}",language.getCode(),language.getClassName());
        }
        return scriptExecutor;
    }

    public ScriptExecutor getScriptExecutor(){
        if (ObjectUtil.isNull(scriptExecutor)){
            ServiceLoader<ScriptExecutor> loader = ServiceLoader.load(ScriptExecutor.class);
            if (loader.iterator().hasNext()){
                try {
                    scriptExecutor = loader.iterator().next().init();
                } catch (ScriptException e) {
                    throw new ScriptSpiException(e.getMessage());
                }
                return scriptExecutor;
            } else {
                throw new ScriptSpiException("script spi component failed to load");
            }
        }
        return scriptExecutor;
    }
}
