package com.tydic.dyc.agr.components.es;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSONObject;
import com.tydic.dyc.agr.service.constants.AgrRspConstant;
import com.tydic.dyc.agr.utils.AgrRu;
import com.tydic.dyc.base.exception.BaseBusinessException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 标题:UocEsSyncManager
 * 说明:es同步管理类
 * 时间:2022/3/1 11:40
 * 作者:田桂银
 */
@Component
@Slf4j
public class AgrEsSyncManager {

    /**
     * 在系统启动时，缓存数据查询实现类
     */
    private final Map<String, IAgrEsSync> manageMap = new ConcurrentHashMap<>();

    @Autowired
    private AgrElasticsearchUtil agrElasticsearchUtil;

    private static final String OBJ_JSON = "objJson";

    /**
     * 描述:保存数据
     *
     * @param obyType 入参
     * @return void
     * @author tgy
     * @date 2022/8/29 14:49
     */
    public void saveData(AgrEsSyncManagerSaveDataReqBo obyType) {
        String indexName = obyType.getIndexName();
        IAgrEsSync interObj = manageMap.get(indexName);
        if (interObj == null) {
            throw new BaseBusinessException(AgrRspConstant.RESP_CODE_ERROR, "索引同步实现类（" + indexName + "）未找到实现类");
        }
        //入参校验
        validateQryReqBo(obyType);
        //数据查询
        IAgrEsSyncQryRspBo qryRspBo = interObj.qryData(AgrRu.js(obyType, IAgrEsSyncQryReqBo.class));
        // 如果不需要同步
        if (!qryRspBo.getSyncFlag()) {
            return;
        }
        //出参校验
        validateRspArg(qryRspBo);
        JSONObject data = qryRspBo.getJsonObj();
//        data.put(OBJ_JSON, data.toJSONString());
        //保存数据
        agrElasticsearchUtil.addJsonData(interObj.indexName(), qryRspBo.getObjId().toString(), data.toJSONString());
    }

    public void deleteData(AgrEsSyncManagerSaveDataReqBo uocEsSyncManagerSaveDataReqBo) {
        String indexName = uocEsSyncManagerSaveDataReqBo.getIndexName();
        //入参校验
        validateQryReqBo(uocEsSyncManagerSaveDataReqBo);
        IAgrEsSync interObj = manageMap.get(indexName + "_delete");
        IAgrEsSyncQryRspBo qryRspBo = interObj.qryData(AgrRu.js(uocEsSyncManagerSaveDataReqBo, IAgrEsSyncQryReqBo.class));
        if (null == qryRspBo || null == qryRspBo.getJsonObj()) {
            throw new BaseBusinessException(AgrRspConstant.RESP_CODE_COMMON_PARAM_FIELD_EMPTY_ERROR, "入参对象属性[json]对象不能为空");
        }
        String endPoint = "/" + indexName + "/_delete_by_query";
        //保存数据
        agrElasticsearchUtil.deleteDataByCondition(endPoint, qryRspBo.getJsonObj());
    }

    public Boolean updateData(AgrEsSyncManagerSaveDataReqBo uocEsSyncManagerSaveDataReqBo) {
        String indexName = uocEsSyncManagerSaveDataReqBo.getIndexName();
        if (null == uocEsSyncManagerSaveDataReqBo.getJsonObj()) {
            throw new BaseBusinessException(AgrRspConstant.RESP_CODE_COMMON_PARAM_FIELD_EMPTY_ERROR, "入参对象属性[json]对象不能为空");
        }
        if (null == uocEsSyncManagerSaveDataReqBo.getObjId()) {
            throw new BaseBusinessException(AgrRspConstant.RESP_CODE_COMMON_PARAM_FIELD_EMPTY_ERROR, "入参对象ID[objId]对象不能为空");
        }
        String endPoint = "/" + indexName + "/_update_by_query";
        //保存数据
        return agrElasticsearchUtil.updateDataById(endPoint, uocEsSyncManagerSaveDataReqBo.getJsonObj(), uocEsSyncManagerSaveDataReqBo.getObjId().toString());
    }

    private void validateQryReqBo(AgrEsSyncManagerSaveDataReqBo reqBo) {
        if (reqBo == null) {
            throw new BaseBusinessException(AgrRspConstant.RESP_CODE_COMMON_PARAM_FIELD_EMPTY_ERROR, "入参对象[AgrEsSyncManagerSaveDataReqBo]不能为空");
        }
        if (ObjectUtil.isEmpty(reqBo.getIndexName())) {
            throw new BaseBusinessException(AgrRspConstant.RESP_CODE_COMMON_PARAM_FIELD_EMPTY_ERROR, "入参对象属性[索引名称]不能为空");
        }
        if (ObjectUtil.isEmpty(reqBo.getAgrId())) {
            throw new BaseBusinessException(AgrRspConstant.RESP_CODE_COMMON_PARAM_FIELD_EMPTY_ERROR, "入参对象属性[协议id]不能为空");
        }
        if (ObjectUtil.isEmpty(reqBo.getObjId())) {
            throw new BaseBusinessException(AgrRspConstant.RESP_CODE_COMMON_PARAM_FIELD_EMPTY_ERROR, "入参对象属性[对象]不能为空");
        }

    }

    private void validateRspArg(IAgrEsSyncQryRspBo reqBo) {
        if (reqBo == null) {
            throw new BaseBusinessException(AgrRspConstant.RESP_CODE_COMMON_PARAM_FIELD_EMPTY_ERROR, "入参对象[IAgrEsSyncQryRspBo]不能为空");
        }
        if (ObjectUtil.isEmpty(reqBo.getObjId())) {
            throw new BaseBusinessException(AgrRspConstant.RESP_CODE_COMMON_PARAM_FIELD_EMPTY_ERROR, "入参对象属性[对象id]不能为空");
        }
        if (ObjectUtil.isEmpty(reqBo.getJsonObj())) {
            throw new BaseBusinessException(AgrRspConstant.RESP_CODE_COMMON_PARAM_FIELD_EMPTY_ERROR, "入参对象属性[json]对象不能为空");
        }
    }


    /**
     * 描述:注册实现类
     *
     * @param intObj 入参
     * @return void
     * @author tgy
     * @date 2022/3/1 12:50
     */
    public void register(IAgrEsSync intObj) {
        String indexName = intObj.indexName();

        Assert.notNull(intObj, "intObj can not be null");
        Assert.notNull(indexName, "indexName can not be null");

        //此处获取到得必须为空，不为空则为重复
        IAgrEsSync intRet = manageMap.get(indexName);
        if (intRet != null) {
            throw new BaseBusinessException(AgrRspConstant.RESP_CODE_ERROR, "不同索引实现类编码(" + indexName + ")出现多个实现类！");
        }
        log.info(AgrEsSyncManager.class.getSimpleName() + ".register注册了不同索引实现类：" + indexName);

        manageMap.put(indexName, intObj);
    }


}


