package com.chinaunicom.qghb.lyhzq.webtier;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.haotian.remote.ProxyXmlWebApplicationContext;
import com.haotian.remote.RemoteInvokeService;
import com.haotian.remote.RemoteInvokeUtils;
import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.ServletRequestDataBinder;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.util.*;

@Controller
@RequestMapping("/")
public class RestController implements ApplicationContextAware {
    private final Logger logger = LoggerFactory.getLogger(this.getClass());
    private Map<String, InvokeInfo> beanStore = new HashMap<String, InvokeInfo>();
    private final ThreadLocal<ObjectMapper> objectMapperThreadLocal = new ThreadLocal<ObjectMapper>() {
        @Override
        protected ObjectMapper initialValue() {
            return new ObjectMapper();
        }
    };
    private RemoteInvokeService remoteInvokeService;

    private  boolean isPrimitiveWrapper(Class<?> paramType) {
        if (isVoidType(paramType)) return true;
        if (isShortType(paramType)) return true;
        if (isIntegerType(paramType)) return true;
        if (isLongType(paramType)) return true;
        if (isBooleanType(paramType)) return true;
        if (isFloatType(paramType)) return true;
        if (isDoubleType(paramType)) return true;
        if (isStringType(paramType)) return true;
        return false;
    }

    private boolean isStringType(Class<?> paramType) {
        return String.class == paramType;
    }

    private boolean isDoubleType(Class<?> paramType) {
        return Double.class == paramType || Double.class.getSimpleName().equalsIgnoreCase(paramType.getName());
    }

    private boolean isFloatType(Class<?> paramType) {
        return Float.class == paramType || Float.class.getSimpleName().equalsIgnoreCase(paramType.getName());
    }

    private boolean isBooleanType(Class<?> paramType) {
        return Boolean.class == paramType || Boolean.class.getSimpleName().equalsIgnoreCase(paramType.getName());
    }

    private boolean isLongType(Class<?> paramType) {
        return Long.class == paramType || Long.class.getSimpleName().equalsIgnoreCase(paramType.getName());
    }

    private boolean isIntegerType(Class<?> paramType) {
        return Integer.class == paramType || "int".equals(paramType.getName());
    }

    private boolean isShortType(Class<?> paramType) {
        return Short.class == paramType || Short.class.getSimpleName().equalsIgnoreCase(paramType.getName());
    }

    private boolean isVoidType(Class<?> paramType) {
        return Void.class == paramType || Void.class.getSimpleName().equalsIgnoreCase(paramType.getName());
    }

    @RequestMapping("invoke")
    @ResponseBody
    public Object invoke(HttpServletRequest request, String service) {
        Object result = null;
        InvokeInfo invokeInfo = beanStore.get(service);
        Method method = invokeInfo.getMethod();
        if (!invokeInfo.isParsed()) {
            String[] parameterNames = this.remoteInvokeService.getParameterNames(invokeInfo.getRemoteInvokeKey());
            if (parameterNames != null && parameterNames.length == invokeInfo.getParamInfoList().size()) {
                List<ParamInfo> paramInfoList = invokeInfo.getParamInfoList();
                Class<?>[] parameterTypes = method.getParameterTypes();
                for (int i = 0; i < parameterNames.length; i++) {
                    ParamInfo paramInfo = paramInfoList.get(i);
                    paramInfo.setParamName(parameterNames[i]);
                    paramInfo.setParamType(parameterTypes[i]);
                }
                invokeInfo.setParsed(true);
            }
        }
        List<ParamInfo> paramInfoList = invokeInfo.getParamInfoList();
        Object[] commands = new Object[paramInfoList.size()];
        boolean isJson = "application/json".equals(request.getContentType()) || "text/json".equals(request.getContentType());
        for (int i = 0; i < commands.length; i++) {
            ParamInfo paramInfo = paramInfoList.get(i);
            Object command = null;
            Class<?> paramType = paramInfo.getParamType();
            if (isPrimitiveWrapper(paramType)) {
                if (isJson) {
                    throw new IllegalArgumentException("Unsupported json invoke for primitive for[" + getInvokeTargetName(invokeInfo) + "]");
                }
                String value = request.getParameter(paramInfo.getParamName());
                if (isIntegerType(paramType)) {
                    command = Integer.valueOf(value);
                } else if (isBooleanType(paramType)) {
                    command = Boolean.valueOf(value);
                } else if (isLongType(paramType)) {
                    command = Long.valueOf(value);
                } else if (isDoubleType(paramType)) {
                    command = Double.valueOf(value);
                } else if (isVoidType(paramType)) {
                    command = null;
                } else if (isFloatType(paramType)) {
                    command = Float.valueOf(value);
                } else if (isShortType(paramType)) {
                    command = Short.valueOf(value);
                } else if (isStringType(paramType)) {
                    command = value;
                } else {
                    throw new IllegalArgumentException("Unsupported parameterType[" + paramType.getName() + "] for [" + getInvokeTargetName(invokeInfo) + "]");
                }
            } else {
                if (isJson) {
                    InputStream jsonInputStream = null;
                    try {
                        jsonInputStream = request.getInputStream();
                        command = objectMapperThreadLocal.get().readValue(jsonInputStream, paramType);
                    } catch (Throwable e) {
                        throw new IllegalStateException("Bind param[" + paramType.getName() + "] for [" + getInvokeTargetName(invokeInfo) + "] error", e);
                    } finally {
                        if (jsonInputStream != null) {
                            try {
                                jsonInputStream.close();
                            } catch (IOException e) {
                                logger.error("Close bind inputstream for param[" + paramType.getName() + "] for [" + getInvokeTargetName(invokeInfo) + "] error", e);
                            }
                        }
                    }
                } else {
                    try {
                        command = paramType.newInstance();
                    } catch (Throwable e) {
                        throw new UnsupportedOperationException("Init parameterType[" + paramType.getName() + "] for [" + getInvokeTargetName(invokeInfo) + "] error", e);
                    }
                    ServletRequestDataBinder binder = new ServletRequestDataBinder(command);
                    binder.bind(request);
                    command = binder.getTarget();
                }
            }
            commands[i] = command;
        }
        try {
            result = method.invoke(invokeInfo.getBean(), commands);
        } catch (Throwable e) {
            throw new RuntimeException("Invoke [" + getInvokeTargetName(invokeInfo) + "] error", e);
        }
        return result;
    }

    private String getInvokeTargetName(InvokeInfo invokeInfo) {
        return invokeInfo.getBean().getClass().getName() + "." + invokeInfo.getMethod().getName();
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        Map<String, Object> consumerStore = new HashMap<String, Object>();
        try {
            Resource[] resources = applicationContext.getResources("classpath*:proxy-remote-consumer.xml");
            for (Resource resource: resources) {
                SAXReader reader = new SAXReader();
                InputStream ins = resource.getInputStream();
                Document document = reader.read(ins);
                ins.close();
                String consumerName = ProxyXmlWebApplicationContext.getRemoteStrategy().equals("DUBBO") ? "reference": "consumer";
                List<Element> consumers = (List<Element>) document.getRootElement().elements(consumerName);
                for (Element consumer: consumers) {
                    String consumerId = consumer.attributeValue("id");
                    String consumerInterface = consumer.attributeValue("interface");
                    String consumerGroup = consumer.attributeValue("group");
                    String consumerVersion = consumer.attributeValue("version");
                    Object consumerBean = applicationContext.getBean(consumerId);
                    if (consumerBean == null) {
                        continue;
                    }
                    if (logger.isDebugEnabled()) {
                        logger.debug("find spring bean[" + consumerInterface + ":" + consumerGroup + ":" + consumerVersion + "]");
                    }
                    consumerStore.put(consumerInterface + ":" + consumerGroup + ":" + consumerVersion, consumerBean);
                }
            }
        } catch (Exception e) {
            throw new RuntimeException("init consumer invoke error:", e);
        }
        Resource resource = applicationContext.getResource("classpath:/init.json");
        ObjectMapper om = new ObjectMapper();
        try {
            InputStream ins = resource.getInputStream();
            JsonNode initJson = om.readTree(ins);
            Iterator<JsonNode> beanList = initJson.elements();
            while (beanList.hasNext()) {
                JsonNode bean = beanList.next();
                String service = bean.get("service").asText();
                String consumerInterface = bean.get("service_name").asText();
                String consumerVersion =  bean.get("service_version").asText();
                String consumerGroup =  bean.get("group_name").asText();
                String consumerMethod = bean.get("method_name").asText();
                Object consumer = consumerStore.get(consumerInterface + ":" + consumerGroup + ":" + consumerVersion);
                if (consumer == null) {
                    throw new IllegalStateException("required invoke bean[" + consumerInterface + ":" + consumerGroup + ":" + consumerVersion + "] required method[" + consumerMethod + "]");
                }
//                Method[] methods = consumer.getClass().getMethods();
                Method[] methods = Class.forName(consumerInterface).getMethods();
                Method targetMethod = null;
                for (Method method: methods) {
                    if (method.getName().equals(consumerMethod)) {
                        targetMethod = method;
                        break;
                    }
                }
                if (targetMethod == null) {
                    throw new IllegalStateException("invoke bean[" + consumerInterface + ":" + consumerGroup + ":" + consumerVersion + "] required method[" + consumerMethod + "]");
                }
                if (logger.isDebugEnabled()) {
                    logger.debug("find invoke bean[" + consumerInterface + ":" + consumerGroup + ":" + consumerVersion + "]");
                }
                InvokeInfo invokeInfo = new InvokeInfo();
                invokeInfo.setBean(consumer);
                invokeInfo.setMethod(targetMethod);
                Class<?>[] parameterTypes = targetMethod.getParameterTypes();
                List<ParamInfo> paramInfoList = new ArrayList<ParamInfo>();
                for (int i = 0; i < parameterTypes.length; i++) {
                    ParamInfo paramInfo = new ParamInfo();
                    paramInfo.setParamType(parameterTypes[i]);
                    paramInfoList.add(paramInfo);
                }
                invokeInfo.setParamInfoList(paramInfoList);
                invokeInfo.setRemoteInvokeKey(RemoteInvokeUtils.getRemoteKey(consumerInterface, consumerMethod, consumerGroup, consumerVersion));
                beanStore.put(service, invokeInfo);
            }
        } catch (Exception e) {
            throw new RuntimeException("init consumer service error:", e);
        }
    }

    public void setRemoteInvokeService(RemoteInvokeService remoteInvokeService) {
        this.remoteInvokeService = remoteInvokeService;
    }
}
