package com.ohaotian.plugin.cache.config;

import com.ohaotian.plugin.cache.CacheClient;
import com.ohaotian.plugin.cache.RedisClusterManager;
import com.ohaotian.plugin.cache.RedisManager;
import com.ohaotian.plugin.cache.impl.CacheClientImpl;
import com.ohaotian.plugin.common.util.RSAUtil;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.pool2.impl.GenericObjectPoolConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Conditional;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import redis.clients.jedis.*;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

@Configuration
public class PluginCacheConfig {

    private static final Logger logger = LoggerFactory.getLogger(PluginCacheConfig.class);

    @Value("${redis.host}")
    private String host;

    @Value("${redis.port}")
    private int port;

    @Value("${redis.timeout}")
    private int timeout;

    @Value("${redis.pwd:}")
    private String pwd;

    @Value("${redis.sentinel.pwd:}")
    private String sentinelPwd;

    @Value("${redis.pwd.publicKey:}")
    private String publicKey;

    @Value("${redis.database:0}")
    private int database;

    @Value("${redis.soTimeOut:2000}")
    private int soTimeOut; //@Version 2.0.7
    @Value("${redis.maxAttemts:5}")
    private int maxAttemts; //@Version 2.0.7

    @Value("${redis.pool.maxTotal}")
    private int poolMaxTotal;

    @Value("${redis.pool.maxIdle}")
    private int poolMaxIdle;

    @Value("${redis.pool.maxWaitMillis}")
    private int poolMaxWaitMillis;

    @Value("${redis.pool.testOnBorrow:false}")
    private boolean poolTestOnBorrow;

    @Value("${redis.pool.testOnReturn:false}")
    private boolean poolTestOnReturn;

    @Value("${redis.masterName:}")
    private String masterName;

    /**
     * 如果有使用加密则执行解密
     */
    public void encrypt() {
        if (StringUtils.isNotEmpty(publicKey)) {
            try {
                logger.info("redis密码启用加密！");
                pwd = RSAUtil.decrypt(publicKey, pwd);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    public static void main(String[] args) {
        try {
            RSAUtil.genKeyPair("123abcd");
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    @Bean
    @Primary
    public JedisPoolConfig jedisPoolConfig() {
        printRedisConfig();
        encrypt();
        JedisPoolConfig config = new JedisPoolConfig();
        config.setMaxTotal(poolMaxTotal);
        config.setMaxIdle(poolMaxIdle);
        config.setMaxWaitMillis(poolMaxWaitMillis);
        config.setTestOnBorrow(poolTestOnBorrow);
        config.setTestOnReturn(poolTestOnReturn);
        return config;
    }

    private void printRedisConfig() {
        logger.info("plugin-cache-config:redis:host:{}, port:{}, timeout:{}, pwd:{}, database:{}", host, port, timeout, pwd, database);
        logger.info("plugin-cache-config:reids:poolMaxTotal:{}, poolMaxIdle:{}, poolMaxWaitMillis:{}, poolTestOnBorrow:{}, poolTestOnReturn:{}", poolMaxTotal, poolMaxIdle, poolMaxWaitMillis, poolTestOnBorrow, poolTestOnReturn);
    }

    /**
     * 单机
     *
     * @return
     */
    @Bean
    @Primary
    @Conditional(SignleOrNot.class)
    public JedisPool jedisPool() {
        JedisPool jedisPool;
        if (StringUtils.isNotEmpty(pwd)) {
            jedisPool = new JedisPool(jedisPoolConfig(), host, port, timeout, pwd, database);
        } else {
            //无密码不支持选择database
            jedisPool = new JedisPool(jedisPoolConfig(), host, port, timeout);
        }
        return jedisPool;
    }

    /**
     * 集群
     *
     * @return
     */
    @Bean
    @Primary
    @Conditional(ClusterOrNot.class)
    public JedisCluster jedisCluster() {
        String clusterNodes = host;
        // 集群redis
        String[] cNodes = clusterNodes.split(",");
        Set<HostAndPort> nodes = new HashSet<>();
        // 分割出集群节点
        for (String node : cNodes) {
            String[] hp = node.split(":");
            nodes.add(new HostAndPort(hp[0], Integer.parseInt(hp[1])));
        }

        GenericObjectPoolConfig poolConfig = new GenericObjectPoolConfig();
        // 设置池配置，例如最大连接数、最大空闲连接数等
        poolConfig.setMaxTotal(poolMaxTotal);
        poolConfig.setMaxIdle(poolMaxIdle);
        // 其他池配置设置...

        JedisCluster jedisCluster;
        if (StringUtils.isNotEmpty(pwd)) {
            jedisCluster = new JedisCluster(nodes, timeout, soTimeOut, maxAttemts, pwd, poolConfig);
        } else {
            jedisCluster = new JedisCluster(nodes, timeout, soTimeOut, maxAttemts, poolConfig);
        }

        return jedisCluster;
    }

    /**
     * 哨兵
     *
     * @return
     */
    @Bean
    @Primary
    @Conditional(SentinelOrNot.class)
    public JedisSentinelPool jedisSentinel() {
        JedisSentinelPool jedisSentinelPool = null;

        String[] cNodes = host.split(",");
        Set<String> sentinels = new HashSet<>(Arrays.asList(cNodes));

        logger.info("jedisSentinel配置：masterName={},sentinels={},pwd={},database={},sentinelPwd={}", masterName, sentinels, pwd, database, sentinelPwd);

        if (StringUtils.isNotBlank(pwd) && StringUtils.isNotBlank(sentinelPwd)) {
            jedisSentinelPool = new JedisSentinelPool(masterName, sentinels, new JedisPoolConfig(), 2000, 2000, pwd, database, (String) null, 2000, 2000, sentinelPwd, (String) null);
        } else if (StringUtils.isNotBlank(pwd)) {
            jedisSentinelPool = new JedisSentinelPool(masterName, sentinels, new JedisPoolConfig(), timeout, pwd);
        } else if (timeout != 0) {
            jedisSentinelPool = new JedisSentinelPool(masterName, sentinels, new JedisPoolConfig(), timeout);
        } else {
            jedisSentinelPool = new JedisSentinelPool(masterName, sentinels, new JedisPoolConfig());
        }
        return jedisSentinelPool;
    }

    @Bean
    @Primary
    @Conditional(ClusterOrNot.class)
    public RedisClusterManager redisClusterManager() {
        RedisClusterManager manager = new RedisClusterManager();
        manager.setJedisClusterPool(jedisCluster());
        manager.setExpire(0);
        return manager;
    }

    @Bean
    @Primary
    @Conditional(SignleOrNot.class)
    public RedisManager redisManager() {
        RedisManager manager = new RedisManager();
        manager.setJedisPool(jedisPool());
        return manager;
    }

    @Bean
    @Primary
    public CacheClient cacheClient() {

        CacheClient cacheClient = new CacheClientImpl();
        if (host.indexOf(",") > -1) {
            logger.info("redis-cluster 启动");
            cacheClient.setCacheManager(redisClusterManager());
        } else {
            logger.info("redis 单机启动");
            cacheClient.setCacheManager(redisManager());
        }
        return cacheClient;
    }
}
