package com.ohaotian.plugin.es.elasticsearch.builder.create;

import com.ohaotian.plugin.es.Const;
import com.ohaotian.plugin.es.builder.create.ColumnsBuilder;
import com.ohaotian.plugin.es.builder.create.CreateRequestBuilder;
import com.ohaotian.plugin.es.builder.create.SettingsBuilder;
import com.ohaotian.plugin.es.builder.schema.DataType;
import com.ohaotian.plugin.es.elasticsearch.ElasticSearchClient;
import java.io.IOException;
import java.util.List;
import org.elasticsearch.action.admin.indices.create.CreateIndexRequestBuilder;
import org.elasticsearch.action.admin.indices.create.CreateIndexResponse;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * @author pengys5
 */
@Component
public class EsCreateRequestBuilder implements CreateRequestBuilder {

    private final Logger logger = LogManager.getLogger(EsCreateRequestBuilder.class);

    @Autowired
    private ElasticSearchClient client;

    @Override public String build(String indexName, SettingsBuilder settingsBuilder, ColumnsBuilder columnsBuilder) {
        CreateIndexRequestBuilder createIndexRequestBuilder = client.createIndexRequestBuilder();
        createIndexRequestBuilder.setIndex(indexName);
        try {
            createIndexRequestBuilder.setSettings(createSettingBuilder(settingsBuilder));
            createIndexRequestBuilder.addMapping(Const.ELASTIC_SEARCH_TYPE, createMappingBuilder(columnsBuilder));
            CreateIndexResponse response = createIndexRequestBuilder.get();
            return response.toString();
        } catch (IOException e) {
            logger.error(e.getMessage(), e);
        }
        return null;
    }

    private XContentBuilder createSettingBuilder(SettingsBuilder settingsBuilder) throws IOException {
        SettingsBuilder.Setting setting = settingsBuilder.build();

        return XContentFactory.jsonBuilder()
            .startObject()
            .field("index.number_of_shards", setting.getNumberOfShards())
            .field("index.number_of_replicas", setting.getNumberOfReplicas())
            .field("index.refresh_interval", String.valueOf(setting.getRefreshInterval()) + "s")
            .endObject();
    }

    private XContentBuilder createMappingBuilder(ColumnsBuilder columnsBuilder) throws IOException {
        XContentBuilder mappingBuilder = XContentFactory.jsonBuilder()
            .startObject()
            .startObject("properties");

        List<ColumnsBuilder.Column> columns = columnsBuilder.build();
        for (ColumnsBuilder.Column column : columns) {
            if (column.getDataType().equals(DataType.Nested)) {
                mappingBuilder
                    .startObject(column.getName())
                    .field("type", column.getDataType().name().toLowerCase())
                    .startObject("properties")
                    .startObject(Const.ELASTIC_NESTED_ID)
                    .field("type", "string".toLowerCase())
                    .endObject()
                    .endObject()
                    .endObject();
            } else if (column.getDataType().equals(DataType.Text)) {
                mappingBuilder
                    .startObject(column.getName())
                    .field("type", "string")
                    .endObject();
            } else {
                mappingBuilder
                    .startObject(column.getName())
                    .field("type", column.getDataType().name().toLowerCase())
                    .endObject();
            }
        }

        mappingBuilder.endObject().endObject();
        logger.debug("create elasticsearch index: {}", mappingBuilder.string());
        return mappingBuilder;
    }
}
