package com.ohaotian.plugin.es.opensearch.builder.search;

import com.aliyun.opensearch.SearcherClient;
import com.aliyun.opensearch.sdk.dependencies.com.google.common.collect.Lists;
import com.aliyun.opensearch.sdk.dependencies.org.json.JSONObject;
import com.aliyun.opensearch.sdk.generated.commons.OpenSearchClientException;
import com.aliyun.opensearch.sdk.generated.commons.OpenSearchException;
import com.aliyun.opensearch.sdk.generated.search.Aggregate;
import com.aliyun.opensearch.sdk.generated.search.Config;
import com.aliyun.opensearch.sdk.generated.search.Order;
import com.aliyun.opensearch.sdk.generated.search.Rank;
import com.aliyun.opensearch.sdk.generated.search.SearchFormat;
import com.aliyun.opensearch.sdk.generated.search.SearchParams;
import com.aliyun.opensearch.sdk.generated.search.Sort;
import com.aliyun.opensearch.sdk.generated.search.SortField;
import com.aliyun.opensearch.sdk.generated.search.general.SearchResult;
import com.aliyun.opensearch.search.SearchResultDebug;
import com.ohaotian.plugin.es.builder.search.AggregateBuilder;
import com.ohaotian.plugin.es.builder.search.FetchBuilder;
import com.ohaotian.plugin.es.builder.search.FilterBuilder;
import com.ohaotian.plugin.es.builder.search.QueryBuilder;
import com.ohaotian.plugin.es.builder.search.RankBuilder;
import com.ohaotian.plugin.es.builder.search.SearchRequestBuilder;
import com.ohaotian.plugin.es.builder.search.SortBuilder;
import com.ohaotian.plugin.es.builder.search.filter.FilterCondition;
import com.ohaotian.plugin.es.builder.search.result.Result;
import com.ohaotian.plugin.es.builder.search.sort.SearchSortMode;
import com.ohaotian.plugin.es.builder.search.sort.SortColumn;
import com.ohaotian.plugin.es.opensearch.OpenSearchClient;
import java.util.List;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * @author pengys5
 */
@Component
public class OsSearchRequestBuilder implements SearchRequestBuilder {

    private final Logger logger = LogManager.getLogger(OsSearchRequestBuilder.class);

    @Autowired
    private OpenSearchClient client;

    @Override
    public Result build(String indexName, int from, int size, QueryBuilder queryBuilder, FilterBuilder filterBuilder,
        FetchBuilder fetchBuilder, SortBuilder sortBuilder, AggregateBuilder aggregateBuilder,RankBuilder rankBuilder,String kvpairs) {
        SearcherClient searchClient = this.client.createSearchClient();

        Config config = new Config(Lists.newArrayList(indexName));
        config.setStart(from);
        config.setKvpairs(kvpairs);

        if (size > 0) {
            config.setHits(size);
        }
        config.setSearchFormat(SearchFormat.JSON);

        if (fetchBuilder != null) {
            config.setFetchFields(fetchBuilder.build());
        }

        SearchParams searchParams = new SearchParams(config);

        if (queryBuilder != null) {
            searchParams.setQuery(OpenSearchQueryBuilder.INSTANCE.build(queryBuilder.build()));
        }

        if (filterBuilder != null) {
            searchParams.setFilter(OpenSearchFilterBuilder.INSTANCE.build(filterBuilder.build()));
        }

        if (sortBuilder != null) {
            Sort sorter = new Sort();
            for (SortColumn column : sortBuilder.build()) {
                if (SearchSortMode.ASC.equals(column.getMode())) {
                    sorter.addToSortFields(new SortField(column.getName(), Order.INCREASE));
                } else if (SearchSortMode.DESC.equals(column.getMode())) {
                    sorter.addToSortFields(new SortField(column.getName(), Order.DECREASE));
                }
            }
            searchParams.setSort(sorter);
        }
        if(rankBuilder !=null){
			// 设置粗精排表达式
			Rank rank = new Rank();
			rank.setFirstRankName(rankBuilder.build().getFirstRankName());
			rank.setSecondRankName(rankBuilder.build().getSecondRankName());
			rank.setReRankSize(rankBuilder.build().getReRankSize());//设置参与精排文档个数
			searchParams.setRank(rank);
        }
        if (aggregateBuilder != null) {
            List<AggregateBuilder.AggregateColumn> aggregateColumns = aggregateBuilder.build();
            for (AggregateBuilder.AggregateColumn aggregateColumn : aggregateColumns) {
                Aggregate aggregate = new Aggregate();
                aggregate.setGroupKey(aggregateColumn.getName());

                StringBuilder aggFuncBuilder = new StringBuilder();
                if (aggregateColumn.isNested()) {
                    aggFuncBuilder.append("count()");
                }

                for (int i = 0; i < aggregateColumn.getFunctions().size(); i++) {
                    AggregateBuilder.AggregateFunction aggregateFunction = aggregateColumn.getFunctions().get(i);
                    if (i > 0) {
                        aggFuncBuilder.append("#");
                    }
                    if (AggregateBuilder.FuncName.SUM.equals(aggregateFunction.getFuncName())) {
                        aggFuncBuilder.append("sum(").append(aggregateFunction.getColumnName()).append(")");
                    }
                    if (AggregateBuilder.FuncName.MAX.equals(aggregateFunction.getFuncName())) {
                        aggFuncBuilder.append("max(").append(aggregateFunction.getColumnName()).append(")");
                    }
                    if (AggregateBuilder.FuncName.MIN.equals(aggregateFunction.getFuncName())) {
                        aggFuncBuilder.append("min(").append(aggregateFunction.getColumnName()).append(")");
                    }
                    if (AggregateBuilder.FuncName.COUNT.equals(aggregateFunction.getFuncName())) {
                        aggFuncBuilder.append("count()");
                    }
                }
                aggregate.setAggFun(aggFuncBuilder.toString());

                FilterCondition filterCondition = aggregateColumn.filterBuilder().build();
                if (filterCondition != null) {
                    aggregate.setAggFilter(OpenSearchFilterBuilder.INSTANCE.build(filterCondition));
                }
                searchParams.addToAggregates(aggregate);
            }
        }

        SearchResult searchResult;
        try {
            logger.debug("search params: {}", searchParams.toString());
            searchResult = searchClient.execute(searchParams);
            String resultJsonStr = searchResult.getResult();
            logger.debug("search result: {}", resultJsonStr);

            JSONObject resultJson = new JSONObject(resultJsonStr);

            Result result = new Result();
            //个别用户可能需要debug请求地址信息
            SearchResultDebug searchdebugrst = searchClient.executeDebug(searchParams);
            //输出上次查询请求串信息
            logger.debug("search url: {}",searchdebugrst.getRequestUrl());
            
            if (resultJson.get("status").equals("OK")) {
                result.setStatus(true);

                JSONObject responseResult = resultJson.getJSONObject("result");
                result.setTotal(responseResult.getLong("viewtotal"));
                result.setItems(responseResult.getJSONArray("items"));
                result.setAggs(responseResult.getJSONArray("facet"));
                
            }else{
            	JSONObject responseResult = resultJson.getJSONObject("result");
                result.setTotal(responseResult.getLong("viewtotal"));
                result.setItems(responseResult.getJSONArray("items"));
                result.setAggs(responseResult.getJSONArray("facet"));
                result.setErrors(resultJson.getJSONArray("errors"));
            }

            return result;
        } catch (OpenSearchException | OpenSearchClientException e) {
            logger.error(e.getMessage(), e);
        }
        return null;
    }
}
