package com.ohaotian.plugin.es.elasticsearch.builder.search;

import com.aliyun.opensearch.sdk.dependencies.org.json.JSONObject;
import com.ohaotian.plugin.es.Const;
import com.ohaotian.plugin.es.builder.search.AggregateBuilder;
import com.ohaotian.plugin.es.builder.search.FetchBuilder;
import com.ohaotian.plugin.es.builder.search.FilterBuilder;
import com.ohaotian.plugin.es.builder.search.QueryBuilder;
import com.ohaotian.plugin.es.builder.search.RankBuilder;
import com.ohaotian.plugin.es.builder.search.SearchRequestBuilder;
import com.ohaotian.plugin.es.builder.search.SortBuilder;
import com.ohaotian.plugin.es.builder.search.query.QueryCondition;
import com.ohaotian.plugin.es.builder.search.result.Result;
import com.ohaotian.plugin.es.builder.search.sort.SearchSortMode;
import com.ohaotian.plugin.es.builder.search.sort.SortColumn;
import com.ohaotian.plugin.es.elasticsearch.ElasticSearchClient;
import java.util.List;
import org.elasticsearch.action.search.SearchResponse;
import org.elasticsearch.search.aggregations.AggregationBuilders;
import org.elasticsearch.search.aggregations.bucket.nested.NestedBuilder;
import org.elasticsearch.search.sort.SortBuilders;
import org.elasticsearch.search.sort.SortOrder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * @author pengys5
 */
@Component
public class EsSearchRequestBuilder implements SearchRequestBuilder {

    private final Logger logger = LogManager.getLogger(EsSearchRequestBuilder.class);

    @Autowired
    private ElasticSearchClient client;

    @Override
    public Result build(String indexName, int from, int size, QueryBuilder queryBuilder, FilterBuilder filterBuilder,
        FetchBuilder fetchBuilder, SortBuilder sortBuilder, AggregateBuilder aggregateBuilder,RankBuilder rankBuilder,String kvpairs) {
        org.elasticsearch.action.search.SearchRequestBuilder searchRequestBuilder = this.client.searchRequestBuilder();
        searchRequestBuilder.setIndices(indexName);
        searchRequestBuilder.setTypes(Const.ELASTIC_SEARCH_TYPE);
        searchRequestBuilder.setFrom(from);

        if (size > 0) {
            searchRequestBuilder.setSize(size);
        }

        if (fetchBuilder != null) {
            String[] includes = new String[fetchBuilder.build().size()];
            searchRequestBuilder.setFetchSource(fetchBuilder.build().toArray(includes), null);
        }

        if (queryBuilder != null) {
            QueryCondition condition = queryBuilder.build();
            searchRequestBuilder.setQuery(ElasticSearchQueryBuilder.INSTANCE.build(condition));
        }

        if (filterBuilder != null) {
            searchRequestBuilder.setPostFilter(ElasticSearchFilterBuilder.INSTANCE.build(filterBuilder.build()));
        }

        if (sortBuilder != null) {
            for (SortColumn column : sortBuilder.build()) {
                if (SearchSortMode.ASC.equals(column.getMode())) {
                    searchRequestBuilder.addSort(SortBuilders.fieldSort(column.getName()).order(SortOrder.ASC));
                } else if (SearchSortMode.DESC.equals(column.getMode())) {
                    searchRequestBuilder.addSort(SortBuilders.fieldSort(column.getName()).order(SortOrder.DESC));
                }
            }
        }

        if (aggregateBuilder != null) {
            List<AggregateBuilder.AggregateColumn> aggColumns = aggregateBuilder.build();
            for (AggregateBuilder.AggregateColumn aggColumn : aggColumns) {
                if (aggColumn.isNested()) {
                    NestedBuilder nestedAggregationBuilder = AggregationBuilders.nested("nested_option").path(aggColumn.getName());
                    nestedAggregationBuilder.subAggregation(AggregationBuilders.terms("group_" + Const.ELASTIC_NESTED_ID).field(aggColumn.getName() + "." + Const.ELASTIC_NESTED_ID));
                    searchRequestBuilder.addAggregation(nestedAggregationBuilder);
                } else {
                    searchRequestBuilder.addAggregation(AggregationBuilders.terms(aggColumn.getName()));
                }
            }
        }

        logger.debug("search request: {}", searchRequestBuilder);
        SearchResponse response = searchRequestBuilder.get();
        String resultJsonStr = response.toString();
        logger.debug("search result: {}", resultJsonStr);

        JSONObject resultJson = new JSONObject(resultJsonStr);

        Result result = new Result();
        result.setStatus(true);

        JSONObject responseResult = resultJson.getJSONObject("hits");
        result.setTotal(responseResult.getLong("total"));
        result.setItems(responseResult.getJSONArray("hits"));
        if(responseResult.has("aggregations")){
            result.setAggs(responseResult.getJSONArray("aggregations"));
        }
        return result;
    }
}
