package com.ohaotian.plugin.file.config;

import com.ohaotian.plugin.file.FileClient;
import com.ohaotian.plugin.file.FileClientFactory;
import com.ohaotian.plugin.file.constant.FileType;
import com.ohaotian.plugin.file.fastdfs.FastdfsConfig;
import com.ohaotian.plugin.file.ftp.FtpConfig;
import com.ohaotian.plugin.file.oss.OssConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

/**
 * 标题：Spring集成plugin-file
 * <p>
 * 说明：plugin-file配置类
 * <br>
 * 时间：2018/07/09<br>
 * 版权：copyright © 2018 www.tydic.com Inc. All rights reserved.
 * </p>
 *
 * @author ZhangCheng
 */
@Configuration
public class PluginFileConfig {

    private static final Logger logger = LoggerFactory.getLogger(PluginFileConfig.class);

    @Value("${plugin.file.type:}")
    private String fileType;

    @Value("${oss.endpoint:}")
    private String endpoint;
    @Value("${oss.accesskey:}")
    private String accesskey;
    @Value("${oss.accessKeySecret:}")
    private String accessKeySecret;
    @Value("${oss.bucketName:}")
    private String bucketName;
    @Value("${oss.accessUrl:}")
    private String accessUrl;

    @Value("${ftp.host:}")
    private String ftpHost;
    @Value("${ftp.user:}")
    private String ftpUser;
    @Value("${ftp.pwd:}")
    private String ftpPwd;
    @Value("${ftp.port:}")
    private Integer ftpPort;
    @Value("${ftp.timeout:}")
    private Integer ftpTimeout;

    @Value("${fastdfs.httpAntiStealToken:}")
    private String httpAntiStealToken;
    @Value("${fastdfs.httpSecretKey:}")
    private String httpSecretKey;
    @Value("${fastdfs.httpTrackerHttpPort:}")
    private Integer httpTrackerHttpPort;
    @Value("${fastdfs.trackerServers:}")
    private String trackerServers;

    @Bean
    public OssConfig ossConfig() {
        logger.info("plugin-file:endpoint:{}, accesskey:{}, accessKeySecret:{}, bucketName:{}, accessUrl:{}", endpoint, accesskey, accessKeySecret, bucketName, accessUrl);
        return new OssConfig(endpoint, accesskey, accessKeySecret, bucketName, accessUrl);
    }

    @Bean
    public FtpConfig ftpConfig() {
        logger.info("plugin-file:ftpHost:{}, ftpUser:{}, ftpPwd:{}, ftpPort:{}, ftpTimeout:{}", ftpHost, ftpUser, ftpPwd, ftpPort, ftpTimeout);
        return new FtpConfig(ftpHost, ftpUser, ftpPwd, ftpPort, ftpTimeout);
    }

    @Bean
    public FastdfsConfig fastdfsConfig() {
        logger.info("plugin-file:httpAntiStealToken:{}, httpSecretKey:{}, httpTrackerHttpPort:{}, trackerServers:{}", httpAntiStealToken, httpSecretKey, httpTrackerHttpPort, trackerServers);
        return new FastdfsConfig(httpAntiStealToken, httpSecretKey, httpTrackerHttpPort, trackerServers);
    }

    @Bean
    public FileClient fileClient() {
        logger.info("plugin-file:fileType:{}",fileType);

        // 根据指定文件类型实例化客户端
        if (FileType.isOss(fileType)){
            if (ossConfig().check()){
                return FileClientFactory.getOssClient(ossConfig());
            }
            throw new IllegalArgumentException("OSS 文件上传属性配置错误，请检查配置文件");
        }
        if (FileType.isFtp(fileType)){
            if (ftpConfig().check()){
                return FileClientFactory.getFtpClient(ftpConfig());
            }
            throw new IllegalArgumentException("FTP 文件上传属性配置错误，请检查配置文件");
        }
        if (FileType.isFastdfs(fileType)){
            if (fastdfsConfig().check()){
                return FileClientFactory.getFastdfsClient(fastdfsConfig());
            }
            throw new IllegalArgumentException("FASTDFS 文件上传属性配置错误，请检查配置文件");
        }

        // 根据配置文件参数实例化客户端，当配置了多个客户端连接的参数时，抛出异常
        FileClient fileClient = null;
        int count = 0;
        if (ossConfig().check()){
            count++;
            fileClient = FileClientFactory.getOssClient(ossConfig());
        }
        if (ftpConfig().check()){
            count++;
            fileClient = FileClientFactory.getFtpClient(ftpConfig());
        }
        if (fastdfsConfig().check()){
            count++;
            fileClient = FileClientFactory.getFastdfsClient(fastdfsConfig());
        }
        if (count == 1){
            return fileClient;
        }
        logger.error("文件上传属性配置错误，请检查配置文件");
        throw new IllegalArgumentException("文件上传属性配置错误，请检查配置文件");
    }
}
