package com.ohaotian.plugin.file;

import com.ohaotian.plugin.file.constant.FileType;

import java.io.File;
import java.io.InputStream;
import java.util.List;

/**
 * 标题：文件插件核心接扣抽象实现类
 * <p>
 * 说明：文件插件核心接扣抽象实现类
 * <br>
 * 时间：2018/07/09<br>
 * 版权：copyright © 2018 www.tydic.com Inc. All rights reserved.
 * </p>
 *
 * @author ZhangCheng
 */
public abstract class AbstractFileClient implements FileClient {

    private static final String SLANT_LINE = "/";

    /**
     * 上传文件通过输入流
     *
     * @param uploadFilePath 文件路径
     * @param fileName       文件名称
     * @param inputStream    输入流
     * @return 文件路径及名称
     */
    @Override
    public abstract String uploadFileByInputStream(String uploadFilePath, String fileName, InputStream inputStream);

    /**
     * 下载为文件通过文件路径及名称
     *
     * @param filePath 文件路径及名称
     * @return 文件
     */
    @Override
    public abstract File downloadToFile(String filePath);

    /**
     * 下载为输入流通过文件路径及名称
     *
     * @param filePath 文件路径及名称
     * @return 输入流
     */
    @Override
    public abstract InputStream downLoadToInputStream(String filePath);

    /**
     * 返回指定前缀的文件名称列表
     *
     * @param prefix 参数prefix是指定返回Object的前缀
     * @return 最多返回100条object名称
     */
    @Override
    public List<String> listObjects(String prefix) {
        FileType fileType = this.getFileType();
        if (fileType == null) {
            throw new IllegalArgumentException("this class [" + this.getClass().getName() + "] method getFileType() is null");
        }
        throw new IllegalArgumentException(fileType.name() + " not support listFiles method");
    }

    /**
     * 返回指定后缀的文件名称列表
     *
     * @param filePath 文件路径及名称
     * @param suffix   后缀
     * @return 最多返回100条object名称
     */
    @Override
    public List<String> listFiles(String filePath, String suffix) {
        FileType fileType = this.getFileType();
        if (fileType == null) {
            throw new IllegalArgumentException("this class [" + this.getClass().getName() + "] method getFileType() is null");
        }
        throw new IllegalArgumentException(fileType.name() + " not support listFiles method");
    }

    /**
     * 获取文件类型
     *
     * @return 文件类型
     */
    protected abstract FileType getFileType();

    /**
     * 根据文件路径和文件名称获取全路径
     *
     * @param uploadFilePath 文件路径
     * @param fileName       文件名称
     * @return 文件路径及名称
     */
    protected final String getFilePath(String uploadFilePath, String fileName) {
        if (!uploadFilePath.endsWith(SLANT_LINE)) {
            uploadFilePath = uploadFilePath + SLANT_LINE;
        }
        return uploadFilePath + fileName;
    }
}
