package com.ohaotian.plugin.file.oss;

import com.ohaotian.plugin.file.AbstractFileClient;
import com.ohaotian.plugin.file.constant.FileType;
import org.springframework.stereotype.Component;

import java.io.File;
import java.io.InputStream;
import java.util.List;

/**
 * 标题：使用OSS进行文件上传与下载
 * <p>
 * 说明：使用OSS进行文件上传与下载
 * <br>
 * 时间：2018/07/09<br>
 * 版权：copyright © 2018 www.tydic.com Inc. All rights reserved.
 * </p>
 *
 * @author ZhangCheng
 */
@Component
public class OssFileClient extends AbstractFileClient {

    private OssConfig ossConfig;

    public OssFileClient(OssConfig ossConfig){
        this.ossConfig = ossConfig;
    }

    /**
     * 上传文件通过输入流
     *
     * @param uploadFilePath 文件路径
     * @param fileName       文件名称
     * @param inputStream    输入流
     * @return 文件路径及名称
     */
    @Override
    public String uploadFileByInputStream(String uploadFilePath, String fileName, InputStream inputStream) {
        String filePath = getFilePath(uploadFilePath,fileName);
        OssUtil.uploadFileToOssByInputStream(filePath, ossConfig, inputStream);
        return filePath;
    }

    /**
     * 下载为文件通过文件路径及名称
     *
     * @param filePath 文件路径及名称
     * @return 文件
     */
    @Override
    public File downloadToFile(String filePath) {
        return OssUtil.downloadFile(filePath, ossConfig);
    }

    /**
     * 下载为输入流通过文件路径及名称
     *
     * @param filePath 文件路径及名称
     * @return 输入流
     */
    @Override
    public InputStream downLoadToInputStream(String filePath) {
        return OssUtil.downloadByInputStream(filePath, ossConfig);
    }

    /**
     * 返回指定前缀的文件名称列表
     *
     * @param prefix 参数prefix是指定返回Object的前缀
     * @return 最多返回100条object名称
     */
    @Override
    public List<String> listObjects(String prefix) {
        return OssUtil.listObjects(ossConfig, prefix);
    }

    /**
     * 获取文件类型
     *
     * @return 文件类型
     */
    @Override
    protected FileType getFileType() {
        return FileType.OSS;
    }
}
