package com.ohaotian.plugin.file.minio;

import com.ohaotian.plugin.base.exception.ZTBusinessException;
import com.ohaotian.plugin.file.AbstractFileClient;
import com.ohaotian.plugin.file.constant.FileType;
import com.ohaotian.plugin.file.util.FileUtils;
import io.minio.MinioClient;
import io.minio.PutObjectOptions;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.StringUtils;

import java.io.File;
import java.io.FileOutputStream;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URLDecoder;
import java.time.LocalDate;
import java.util.*;

/**
 * 标题：使用Minio进行文件上传与下载（支持多节点）
 * <p>
 * 说明：使用Minio进行文件上传与下载，支持配置多个端点
 * <br>
 * 时间：2023年12月1日10:18:41
 * </p>
 *
 * @author gujb
 */
@Slf4j
public class MinioFileClient extends AbstractFileClient {

    @Value("${plugin.file.blackType:exe;bat;js;html;dll;jsp;zip;rar}")
    private String fileUploadBlackType;

    private MinioConfig minioConfig;

    public MinioFileClient() {

    }

    public MinioFileClient(MinioConfig minioConfig) {
        this.minioConfig = minioConfig;
    }

    /**
     * 获取支持多节点的MinIO客户端
     */
    private MinioClient getMinioClient() {
        List<String> endpoints = parseEndpoints(minioConfig.getEndPoint());

        if (endpoints.isEmpty()) {
            throw new ZTBusinessException("未配置MinIO端点");
        }

        // 复制端点列表并打乱顺序
        List<String> shuffledEndpoints = new ArrayList<>(endpoints);
        Collections.shuffle(shuffledEndpoints, new Random());

        Exception lastException = null;

        // 尝试连接每个端点
        for (String endpoint : shuffledEndpoints) {
            try {
                MinioClient client = new MinioClient(
                        endpoint,
                        minioConfig.getAccessKeyId(),
                        minioConfig.getAccessKeySecret()
                );

                // 验证连接是否成功
                client.listBuckets();
                return client;
            } catch (Exception e) {
                lastException = e;
                log.warn("连接MinIO端点失败: {}, 尝试下一个", endpoint, e);
            }
        }

        // 所有端点都失败
        assert lastException != null;
        throw new ZTBusinessException("无法连接到任何MinIO端点：" + lastException.getMessage());
    }

    /**
     * 解析多节点配置
     */
    private List<String> parseEndpoints(String endpointConfig) {
        List<String> endpoints = new ArrayList<>();
        if (endpointConfig != null) {
            String[] parts = endpointConfig.split(",");
            for (String part : parts) {
                String endpoint = part.trim();
                if (!endpoint.isEmpty()) {
                    endpoints.add(endpoint);
                }
            }
        }
        return endpoints;
    }

    /**
     * Minio文件上传
     *
     * @param uploadFilePath 存储路径
     * @param fileName       文件名称
     * @param inputStream    输入流
     * @return
     */
    @Override
    public String uploadFileByInputStream(String uploadFilePath, String fileName, InputStream inputStream) {
        return uploadFileByInputStream(uploadFilePath, fileName, inputStream, null);
    }

    public String uploadFileByInputStream(String uploadFilePath, String fileName, InputStream inputStream, String accessAuth) {
        String fileExtName = FileUtils.ext(fileName);
        if (!StringUtils.hasText(fileExtName)) {
            throw new ZTBusinessException("禁止上传无类型文件，请更换后重新上传");
        }
        if (fileUploadBlackType.contains(fileExtName) && !"mp4".equals(fileExtName) && !"json".equals(fileExtName) && !"txt".equals(fileExtName) && !"p12".equals(fileExtName) && !"pfx".equals(fileExtName) && !"cer".equals(fileExtName)) {
            throw new ZTBusinessException("禁止上传该文件，请更换后重新上传");
        }

        // 构建日期目录结构
        LocalDate currentDate = LocalDate.now();
        UUID uuid = UUID.randomUUID();

        String datePath = String.format("%d/%02d/%02d/%s/",
                currentDate.getYear(),
                currentDate.getMonthValue(),
                currentDate.getDayOfMonth(),
                uuid.toString()
        );

        String objectName = "";
        if (uploadFilePath.endsWith("/") || fileName.startsWith("/")) {
            objectName = uploadFilePath + datePath + fileName;
        } else {
            objectName = uploadFilePath + "/" + datePath + fileName;
        }

        try {
            MinioClient minioClient = getMinioClient();
            long partSize = 5 * 1024 * 1024; // 5MB
            PutObjectOptions options = new PutObjectOptions(-1, partSize);
            minioClient.putObject(minioConfig.getBucketName(), objectName, inputStream, options);

            String url = minioClient.getObjectUrl(minioConfig.getBucketName(), objectName);
            String path = keepAfterFourthSlash(url);
            String decodedPath = URLDecoder.decode(path, "UTF-8");
            return decodedPath;
        } catch (Exception e) {
            e.printStackTrace();
            log.error("Minio文件上传失败," + e.getMessage());
        }
        return null;
    }

    public static String keepAfterFourthSlash(String url) {
        int count = 0;
        int index = 0;
        for (int i = 0; i < url.length(); i++) {
            if (url.charAt(i) == '/') {
                count++;
                if (count == 4) {
                    index = i;
                    break;
                }
            }
        }
        return url.substring(index + 1); // 保留第四个斜杠后的内容
    }

    @Override
    public File downloadToFile(String filePath) {
        String fileName = filePath.substring(filePath.lastIndexOf("/"), filePath.length());
        String dir = System.getProperty("java.io.tmpdir");
        if (!dir.endsWith("/")) {
            fileName = dir + File.separator + fileName;
        } else {
            fileName = dir + fileName;
        }
        File dirFile = new File(dir);
        if (!dirFile.exists()) {
            dirFile.mkdirs();
        }
        dirFile = new File(fileName);

        try (InputStream inputStream = this.downLoadToInputStream(filePath);
             OutputStream outputStream = new FileOutputStream(dirFile)) {

            // 将输入流中的内容写入文件
            byte[] buffer = new byte[1024];
            int bytesRead;
            while ((bytesRead = inputStream.read(buffer)) != -1) {
                outputStream.write(buffer, 0, bytesRead);
            }

            return dirFile;
        } catch (Exception e) {
            e.printStackTrace();
            log.error("Minio文件下载失败,filePath=" + filePath + ",异常=" + e.getMessage());
            return null;
        }
    }

    @Override
    public InputStream downLoadToInputStream(String filePath) {
        try {
            MinioClient minioClient = getMinioClient();
            return minioClient.getObject(minioConfig.getBucketName(), filePath);
        } catch (Exception e) {
            e.printStackTrace();
            log.error("Minio文件流下载失败,filePath=" + filePath + ",异常=" + e.getMessage());
            return null;
        }
    }

    @Override
    protected FileType getFileType() {
        return FileType.MINIO;
    }
}