package com.tydic.async.exterior.processer;

import com.alibaba.fastjson.JSONObject;
import com.ohaotian.plugin.base.bo.RspBaseBO;
import com.tydic.async.call.annotation.AsyncInvoker;
import com.tydic.async.call.bo.AsyncRequest;
import com.tydic.async.call.bo.MqMethodMeta;
import com.tydic.async.call.future.DefaultAsyncFuture;
import com.tydic.async.exterior.config.mq.ExteriorMqProvider;
import com.tydic.async.exterior.properties.ExteriorCallProperties;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;

import java.lang.reflect.Field;
import java.lang.reflect.Proxy;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;

/**
 * 标题：类名称:AsyncInvokerBeanProcessor
 * 说明：描述一下类的作用TODO
 * 时间：2022/1/10 3:05 PM
 * 作者 @author hegy2017
 */
@Slf4j
@Component
public class AsyncInvokerBeanProcessor implements BeanPostProcessor {

    /**
     * 缓存生成的动态代理对象,用于多个Controller注入同一类型对象时使用.
     */
    private final ConcurrentMap<String, Object> proxyMap = new ConcurrentHashMap<>();

    @Autowired
    private ExteriorMqProvider exteriorMqProvider;

    @Autowired
    private ExteriorCallProperties exteriorProperties;

    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        //获取该实例中的有@AsyncInvoker注解的field
        Field[] fields = bean.getClass().getDeclaredFields();
        for (Field field : fields) {
            try {
                if (!field.isAccessible()) {
                    field.setAccessible(true);
                }
                AsyncInvoker asyncInvoker = field.getAnnotation(AsyncInvoker.class);
                if (asyncInvoker != null) {
                    //创建代理对象,赋值给该field
                    Object value = createProxy(field.getType());
                    if (value != null) {
                        field.set(bean, value);
                    }
                }
            } catch (Throwable e) {
                log.error("Failed to init remote mq service at filed " + field.getName() + " in class " + bean.getClass().getName() + ", cause: " + e.getMessage(), e);
            }
        }
        return bean;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }

    /**
     * 创建真实代理
     *
     * @param clz
     * @return
     */
    private Object createProxy(Class clz) {

        String interfaceName;

        if (clz.isInterface()) {
            interfaceName = clz.getName();
        } else {
            throw new IllegalStateException("The @MqInvoker property type " + clz.getName() + " is not a interface.");
        }

        Object proxy = proxyMap.get(interfaceName);
        if (proxy == null) {
            Object newProxy = Proxy.newProxyInstance(Thread.currentThread().getContextClassLoader(), new Class[]{clz}, (proxy1, method, args) -> {

                if(log.isDebugEnabled()){
                    log.debug("执行动态代理! method:{} ,args: {}", method, args);
                }
                if (method.getParameters().length != 1) {
                    throw new IllegalAccessException("MQ Service 目前仅支持单参数Object类型方法");
                }
                //动态代理中创建mq传输对象并发送.
                AsyncRequest request = new AsyncRequest();
                if (!StringUtils.isEmpty(exteriorProperties.getTimeout())) {
                    request.setTimeOut(exteriorProperties.getTimeout());
                }
                MqMethodMeta mqMethodMeta = new MqMethodMeta();
                mqMethodMeta.setInterfaceName(clz.getName());
                mqMethodMeta.setMethodName(method.getName());
                mqMethodMeta.setArgs(args);

                final Class<?> returnType = method.getReturnType();

                String[] paramTypeNames = new String[args.length];
                for (int i = 0; i < args.length; i++) {
                    paramTypeNames[i] = args[i].getClass().getName();
                }
                mqMethodMeta.setParamTypeNames(paramTypeNames);
                mqMethodMeta.setAsyncRequest(request);
                //发送mq消息
                exteriorMqProvider.provideMessage(mqMethodMeta);

                //获取mq返回消息
                DefaultAsyncFuture future = DefaultAsyncFuture.newFuture(request);
                try {
                    //开始执行
                    DefaultAsyncFuture.sent(request);
                    //超时获取
                    future.get(request.getTimeOut(), TimeUnit.MILLISECONDS);
                }catch (Exception e){
                    log.error("获取返回信息异常"+e.getMessage());
                    future.cancel();
                    RspBaseBO  rspBaseBO = new RspBaseBO();
                    rspBaseBO.setCode("1");
                    rspBaseBO.setMessage(e.getMessage());
                    future.setResult(rspBaseBO);
                }
                return JSONObject.parseObject(String.valueOf(future.getResult()), returnType);
            });
            proxyMap.putIfAbsent(interfaceName, newProxy);
            proxy = proxyMap.get(interfaceName);
        }
        return proxy;
    }

}
