package com.tydic.async.exterior.webclient;

import com.alibaba.fastjson.JSONObject;
import com.tydic.async.call.bo.AsyncRequest;
import com.tydic.async.call.bo.CallReqContext;
import com.tydic.async.call.constant.AsyncConstant;
import com.tydic.async.call.future.DefaultAsyncFuture;
import com.tydic.async.exterior.config.mq.ExteriorMqWebClientProvider;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.stereotype.Component;

import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import java.io.IOException;
import java.util.Enumeration;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * 标题：类名称:WebclientService
 * 说明：提供给外网调用的服务，支持web调用的方式
 * 时间：2022/4/19 5:13 PM
 * 作者 @author hegy2017
 */
@Component
@Slf4j
public class WebclientService {

    @Autowired
    private ExteriorMqWebClientProvider exteriorMqWebClientProvider;

    /**
     * 调用服务
     *
     * @param request
     * @param reqContext
     * @return
     */
    public Object doService(HttpServletRequest request, CallReqContext reqContext) {
        AsyncRequest asyncRequest = new AsyncRequest();
        reqContext.setMessageId(asyncRequest.getId());
        initReqContext(request, reqContext);
        exteriorMqWebClientProvider.provideMessage(reqContext);
        //获取mq返回消息
        DefaultAsyncFuture future = DefaultAsyncFuture.newFuture(asyncRequest);
        try {
            //开始执行
            DefaultAsyncFuture.sent(asyncRequest);
            //超时获取
            future.get(asyncRequest.getTimeOut(), TimeUnit.MILLISECONDS);
        } catch (Exception e) {
            log.error("future callback but catch this exception; the  message detail is " + e.getMessage());
            future.cancel();
        }
        return JSONObject.parseObject(String.valueOf(future.getResult()));
    }

    /**
     * 初始化请求内容
     *
     * @param request
     * @return
     */
    private void initReqContext(final HttpServletRequest request, CallReqContext reqContext) {

        // 初始化请求信息
        final String method = request.getMethod();
        if (StringUtils.isEmpty(reqContext.getMethod())) {
            reqContext.setMethod(method);
        }
        reqContext.setReqData(readAsBytes(request));
        reqContext.getReqObj(reqContext);
        // 请求头信息
        Map<String, String> headersMap = new ConcurrentHashMap<>(16);
        Enumeration headerNames = request.getHeaderNames();
        while (headerNames.hasMoreElements()) {
            String headerName = (String) headerNames.nextElement();
            headersMap.put(headerName, request.getHeader(headerName));
        }
        //如果没有head头增加默认content-type
        if (StringUtils.isEmpty(headersMap.get(HttpHeaders.CONTENT_TYPE))) {
            headersMap.put(HttpHeaders.CONTENT_TYPE, AsyncConstant.ContentType.APPLICATION_JSON);
        }

        reqContext.setHeadersMap(headersMap);
        final String ip = request.getRemoteAddr();
        reqContext.setFromIp(ip);
        reqContext.setReqAcceptStartTime(System.currentTimeMillis());
    }

    /**
     * 获取请求数据
     *
     * @param request
     * @return
     */
    private static byte[] readAsBytes(HttpServletRequest request) {
        int len = request.getContentLength();
        byte[] buffer = new byte[len];
        ServletInputStream in = null;
        try {
            in = request.getInputStream();
            in.read(buffer, 0, len);
            in.close();
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (null != in) {
                try {
                    in.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return buffer;
    }
}
