package com.tydic.dyc.oc.components.es;

import lombok.extern.slf4j.Slf4j;
import org.apache.http.HttpHost;
import org.apache.http.auth.AuthScope;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.nio.client.HttpAsyncClientBuilder;
import org.apache.http.impl.nio.reactor.IOReactorConfig;
import org.elasticsearch.client.Node;
import org.elasticsearch.client.RestClient;
import org.elasticsearch.client.RestClientBuilder;
import org.elasticsearch.client.RestHighLevelClient;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.FactoryBean;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * 标题：http方式的客户端配置
 * <p>
 * 说明：http方式的客户端配置，配置9200端口的连接方式
 * <br>
 * 时间：2019/1/1716:40<br>
 * </p>
 *
 * @author hegy
 */
@Configuration
@Slf4j
public class UocElasticsearchConfiguration implements FactoryBean<RestHighLevelClient>, InitializingBean, DisposableBean {
    /**
     * 地址
     */
    @Value("${es.cluster.address}")
    private String clusterNodes;

    @Value("${es.source}")
    private String esSource;

    @Value(("${es.client.username}"))
    private String esClientUsername;

    @Value(("${es.client.password}"))
    private String esClientPassword;

    /**
     * 连接数
     */
    @Value(("${es.pool.maxTotal}"))
    private int maxTotal;
    /*获取连接超时时间**/
    @Value(("${es.pool.connectionTimeout}"))
    private int connectionTimeout;
    /**
     * 套接字超时超时时间
     */
    @Value(("${es.pool.socketTimeout}"))
    private int socketTimeout;

    /*最大重试超时**/
    @Value(("${es.pool.maxRetryTimeoutMillis}"))
    private int maxRetryTimeoutMillis;

    @Value(("${es.pool.connectionRequestTimeout}"))
    private int connectionRequestTimeout;


    private RestHighLevelClient restHighLevelClient;

    /**
     * 控制Bean的实例化过程
     *
     * @return
     * @throws Exception
     */
    @Override
    public RestHighLevelClient getObject() throws Exception {
        return restHighLevelClient;
    }

    /**
     * 获取接口返回的实例的class
     *
     * @return
     */
    @Override
    public Class<?> getObjectType() {
        return RestHighLevelClient.class;
    }

    @Override
    public void destroy() {
        try {
            if (restHighLevelClient != null) {
                restHighLevelClient.close();
            }
        } catch (final Exception e) {
            log.error("Error closing ElasticSearch client: ", e);
        }
    }

    @Override
    public boolean isSingleton() {
        return false;
    }

    @Override
    public void afterPropertiesSet() {
        restHighLevelClient = buildClient();
    }

    private RestHighLevelClient buildClient() {

        RestClientBuilder clientBuilder = null;
        try {
            if (StringUtils.isEmpty(clusterNodes)) {
                return null;
            }
            List<String> ipAdrrs = new ArrayList<>(Arrays.asList(clusterNodes.split(",")));
            HttpHost httpHost = null;
            HttpHost[] httpHosts = new HttpHost[ipAdrrs.size()];
            for (int i = 0; i < ipAdrrs.size(); i++) {
                httpHost = new HttpHost(ipAdrrs.get(i).split(":")[0],
                        Integer.valueOf(ipAdrrs.get(i).split(":")[1]),
                        "http");
                httpHosts[i] = (httpHost);
            }
            clientBuilder = RestClient.builder(
                    httpHosts);

    /*        clientBuilder =RestClient.builder(
                new HttpHost(
                        clusterNodes.split(":")[0],
                        9200,
                        "http"));*/
        } catch (Exception e) {
            e.printStackTrace();
        }
        // 设置超时时间
        //clientBuilder.(maxRetryTimeoutMillis);


        // 设置监听器，每次节点失败都可以监听到，可以作额外处理
        clientBuilder.setFailureListener(new RestClient.FailureListener() {
            public void onFailure(HttpHost host) {
                super.onFailure(new Node(host));
                System.out.println(host.getHostName() + "==节点失败了");
            }
        });

        /* 配置异步请求的线程数量，Apache Http Async Client默认启动一个调度程序线程，以及由连接管理器使用的许多工作线程
        （与本地检测到的处理器数量一样多，取决于Runtime.getRuntime().availableProcessors()返回的数量）。线程数可以修改如下,
        这里是修改为1个线程，即默认情况
        */
        clientBuilder.setHttpClientConfigCallback(new RestClientBuilder.HttpClientConfigCallback() {
            @Override
            public HttpAsyncClientBuilder customizeHttpClient(HttpAsyncClientBuilder httpAsyncClientBuilder) {
                return httpAsyncClientBuilder.setDefaultIOReactorConfig(
                        IOReactorConfig.custom().setIoThreadCount(maxTotal).build()
                );
            }
        });

        /*
    配置请求超时，将连接超时（默认为1秒）和套接字超时（默认为30秒）增加，
    这里配置完应该相应地调整最大重试超时（默认为30秒），即上面的setMaxRetryTimeoutMillis，一般于最大的那个值一致即60000
    */
        clientBuilder.setRequestConfigCallback(new RestClientBuilder.RequestConfigCallback() {
            @Override
            public RequestConfig.Builder customizeRequestConfig(RequestConfig.Builder requestConfigBuilder) {
                // 连接5秒超时，套接字连接60s超时
                return requestConfigBuilder.setConnectTimeout(connectionTimeout).setSocketTimeout(socketTimeout).setConnectionRequestTimeout(connectionRequestTimeout);
            }
        });

        /*
    如果ES设置了密码，那这里也提供了一个基本的认证机制，下面设置了ES需要基本身份验证的默认凭据提供程序
    */
        if ("AliYun".equals(esSource)) {
            final CredentialsProvider credentialsProvider = new BasicCredentialsProvider();
            credentialsProvider.setCredentials(AuthScope.ANY,
                    new UsernamePasswordCredentials(esClientUsername, esClientPassword));
            clientBuilder.setHttpClientConfigCallback(new RestClientBuilder.HttpClientConfigCallback() {
                @Override
                public HttpAsyncClientBuilder customizeHttpClient(HttpAsyncClientBuilder httpClientBuilder) {
                    return httpClientBuilder.setDefaultCredentialsProvider(credentialsProvider);
                }
            });
        }


        /*
            上面采用异步机制实现抢先认证，这个功能也可以禁用，这意味着每个请求都将在没有授权标头的情况下发送，然后查看它是否被接受，
         并且在收到HTTP 401响应后，它再使用基本认证头重新发送完全相同的请求，这个可能是基于安全、性能的考虑
         */
    /*    clientBuilder.setHttpClientConfigCallback(new RestClientBuilder.HttpClientConfigCallback() {
            @Override
            public HttpAsyncClientBuilder customizeHttpClient(HttpAsyncClientBuilder httpClientBuilder) {
                // 禁用抢先认证的方式
                httpClientBuilder.disableAuthCaching();
                return httpClientBuilder.setDefaultCredentialsProvider(credentialsProvider);
            }
        });*/

        restHighLevelClient = new RestHighLevelClient(clientBuilder);
        return restHighLevelClient;
    }
}

