package com.tydic.dyc.oc.service.bargaining;

import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.oc.constants.UocConstant;
import com.tydic.dyc.oc.constants.UocRspConstants;
import com.tydic.dyc.oc.constants.UocStateConstants;
import com.tydic.dyc.oc.model.bargaining.IUocBargainingModel;
import com.tydic.dyc.oc.model.bargaining.UocBargainingDo;
import com.tydic.dyc.oc.model.bargaining.UocBargainingProgressRecordDo;
import com.tydic.dyc.oc.model.bargaining.UocBargainingQuotationDo;
import com.tydic.dyc.oc.model.bargaining.qrybo.UocBargainingItemQryBo;
import com.tydic.dyc.oc.model.bargaining.sub.UocBargainingItem;
import com.tydic.dyc.oc.model.bargaining.sub.UocBargainingQuotationItem;
import com.tydic.dyc.oc.service.bargaining.bo.UocCancelBargainingReqBo;
import com.tydic.dyc.oc.service.bargaining.bo.UocRebargainingReqBo;
import com.tydic.dyc.oc.service.bargaining.bo.UocRebargainingRspBo;
import com.tydic.dyc.oc.utils.IdUtil;
import com.tydic.dyc.oc.utils.UocRu;
import lombok.extern.ohaotian.HTServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * 描述： 重新发起议价业务实现
 * @author DUM
 * @date 2023/8/8
 * @Version 1.0
 */
@HTServiceImpl
@Slf4j
public class UocRebargainingServiceImpl implements UocRebargainingService{
    @Autowired
    private IUocBargainingModel uocBargainingModel;
    @Override
    public UocRebargainingRspBo executeRebargaining(UocRebargainingReqBo uocRebargainingReqBo) {
        //1. 参数校验
        this.verifyParam(uocRebargainingReqBo);
        //2. 根据id构造议价及议价列表
        UocBargainingDo bargainingDo =buildBargainingDo(uocRebargainingReqBo);
        //3. 前置状态校验,
       this.verifyDo(bargainingDo);
        //4. 创建报价单
        this.createBargainingQuotation(bargainingDo);
        //5 .修改状态、轮次
        this.updateBargainingStateAndRound(bargainingDo);
        //6. 记录操作动作
        this.insertProgressRecord(uocRebargainingReqBo);
        //7. 返回结果
        return UocRu.success(UocRebargainingRspBo.class);
    }
    /**
     * 参数校验
     * @param bargainingDo
     */
    private UocBargainingDo verifyDo(UocBargainingDo bargainingDo){
        if(!UocStateConstants.BARGAINING_STATES.YJ_YBJ.equals(bargainingDo.getBargainingState())){
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "不能发起重新报价，议价单状态非已报价！");
        }
        if(bargainingDo.getQuoteMaxRound().compareTo(bargainingDo.getQuoteCurrentRound())<0){
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "重新议价次数超出允许的最大次数："+bargainingDo.getQuoteMaxRound()+"次！");
        }
        return bargainingDo;
    }
    /**
     * 参数校验
     * @param reqBO
     */
    private UocBargainingDo buildBargainingDo(UocRebargainingReqBo reqBO){
        UocBargainingDo uocBargainingDo = new UocBargainingDo();
        uocBargainingDo.setBargainingId(reqBO.getBargainingId());
        UocBargainingDo bargainingDo = uocBargainingModel.qryBargainingBy(uocBargainingDo);
        bargainingDo.setUpdateOperId(String.valueOf(reqBO.getUserId()));
        bargainingDo.setUpdateOperName(reqBO.getName());
        return bargainingDo;
    }
    /**
     * 参数校验
     * @param reqBO
     */
    private void verifyParam(UocRebargainingReqBo reqBO){
        if(null == reqBO){
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "入参对象为空");
        }
        if(null == reqBO.getUserId()){
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "当前操作人id不能为空");
        }
        if(StringUtils.isEmpty(reqBO.getName())){
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "当前操作人名称不能为空");
        }
        if(null == reqBO.getBargainingId()){
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "议价单id不能为空");
        }
    }
    /**
     * 更新议价单状态,数量
     * @param uocBargainingDo
     */
    private void updateBargainingStateAndRound(UocBargainingDo uocBargainingDo){
        UocBargainingDo buargainingDo = new UocBargainingDo();
        buargainingDo.setUpdateTime(new Date());
        buargainingDo.setUpdateOperId(String.valueOf(uocBargainingDo.getUpdateOperId()));
        buargainingDo.setUpdateOperName(uocBargainingDo.getUpdateOperName());
        buargainingDo.setBargainingState(UocStateConstants.BARGAINING_STATES.YJ_BJZ);
        buargainingDo.setBargainingId(uocBargainingDo.getBargainingId());
        buargainingDo.setQuoteCurrentRound(uocBargainingDo.getQuoteCurrentRound()+1);
        this.uocBargainingModel.updateBargaining(buargainingDo);
    }
    /**
     * 报价单初始化
     * @param bargainingDo
     */
    private void createBargainingQuotation(UocBargainingDo bargainingDo){
        UocBargainingQuotationDo uocBargainingQuotationDo = new UocBargainingQuotationDo();
        uocBargainingQuotationDo.setQuotationId(IdUtil.nextId());
        uocBargainingQuotationDo.setBargainingId(bargainingDo.getBargainingId());
        uocBargainingQuotationDo.setCreateTime(new Date());
        uocBargainingQuotationDo.setQuotationExpiryTime(bargainingDo.getQuotationExpiryTime());
        uocBargainingQuotationDo.setQuotationRound(bargainingDo.getQuoteCurrentRound()+1);
        uocBargainingQuotationDo.setTotalOriginalPrice(bargainingDo.getTotalPrice());
        uocBargainingQuotationDo.setQuotationStatus("0");
        uocBargainingQuotationDo.setCreateOperId(Long.valueOf(bargainingDo.getBargainingOperId()));
        uocBargainingQuotationDo.setCreateOperName(bargainingDo.getBargainingOperName());
        UocBargainingItemQryBo uocBargainingItemQryBo = new UocBargainingItemQryBo();
            uocBargainingItemQryBo.setBargainingId(bargainingDo.getBargainingId());
        List<UocBargainingItem> bargainingItems = uocBargainingModel.qryBargainingItemList(uocBargainingItemQryBo);
        List<UocBargainingQuotationItem> bargainingQuotationItems = new ArrayList<>();
        for (UocBargainingItem uocBargainingItem:bargainingItems) {
            UocBargainingQuotationItem quotationItem = new UocBargainingQuotationItem();
            quotationItem.setQuotationItemId(IdUtil.nextId());
            quotationItem.setQuotationId(uocBargainingQuotationDo.getQuotationId());
            quotationItem.setBargainingItemId(uocBargainingItem.getBargainingItemId());
            quotationItem.setSkuId(uocBargainingItem.getSkuId());
            bargainingQuotationItems.add(quotationItem);
        }
        uocBargainingQuotationDo.setBargainingQuotationItems(bargainingQuotationItems);
        uocBargainingModel.createQuotation(uocBargainingQuotationDo);
    }
    /**
     * 记录操作动作
     * @param reqBO
     */
    private void insertProgressRecord(UocRebargainingReqBo reqBO){
        //记录操作记录
        UocBargainingProgressRecordDo recordDo = new UocBargainingProgressRecordDo();
        recordDo.setProgressRecordId(IdUtil.nextId());
        recordDo.setBargainingId(reqBO.getBargainingId());
        recordDo.setOperType(UocConstant.BARGAINING_OPER_TYPE.RE_QUOTATION);
        recordDo.setOperTypeName("重新议价");
        recordDo.setOperId(Long.valueOf(reqBO.getUserId()));
        recordDo.setOperName(reqBO.getName());
        recordDo.setOperTime(new Date());
        uocBargainingModel.insertProgressRecord(recordDo);
    }
}
