package com.tydic.dyc.oc.service.domainservice;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.tydic.dyc.base.constants.BaseRspConstant;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.oc.constants.*;
import com.tydic.dyc.oc.model.order.IUocOrderModel;
import com.tydic.dyc.oc.model.order.qrybo.UocConfEffectiveQryBo;
import com.tydic.dyc.oc.model.order.sub.UocConfEffective;
import com.tydic.dyc.oc.model.order.sub.UocOrderAccessory;
import com.tydic.dyc.oc.model.saleorder.IUocSaleOrderModel;
import com.tydic.dyc.oc.model.saleorder.UocSaleOrderDo;
import com.tydic.dyc.oc.model.saleorder.qrybo.UocSaleOrderItemQryBo;
import com.tydic.dyc.oc.model.saleorder.qrybo.UocSaleOrderQryBo;
import com.tydic.dyc.oc.model.saleorder.sub.UocSaleOrderItem;
import com.tydic.dyc.oc.model.shiporder.IUocShipOrderModel;
import com.tydic.dyc.oc.model.shiporder.UocShipOrderDo;
import com.tydic.dyc.oc.service.domainservice.bo.UocSaleShipOrderReqBo;
import com.tydic.dyc.oc.service.domainservice.bo.UocSaleShipOrderReqItemBo;
import com.tydic.dyc.oc.service.domainservice.bo.UocSaleShipOrderRspBo;
import com.tydic.dyc.oc.utils.IdUtil;
import lombok.extern.ohaotian.HTServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 描述: 〈领域服务-发货（基于销售单）实现〉
 * @author : yuhao
 * @date : 2022/8/23 17:00
 */
@HTServiceImpl
public class UocSaleShipOrderServiceImpl implements UocSaleShipOrderService {

    @Autowired
    private IUocShipOrderModel iUocShipOrderModel;
    @Autowired
    private IUocSaleOrderModel iUocSaleOrderModel;
    @Autowired
    private IUocOrderModel iUocOrderModel;
    /**
     * 已发完货
     */
    private static final String OP_FLAG_YES = "1";
    /**
     * 未发完货
     */
    private static final String OP_FLAG_NO = "0";
    /**
     * 订单来源-电商
     */
    private static final String EC_IMPORT = "2";
    /**
     * 全部-供应商ID
     */
    private static final String SUP_NO = "0";

    @Override
    public UocSaleShipOrderRspBo dealImplShipOrder(UocSaleShipOrderReqBo reqBo) {
        //参数校验
        this.validateArg(reqBo);
        //获取销售单信息
        UocSaleOrderDo saleOrderInfo = this.getSaleOrderInfo(reqBo);
        //获取销售单明细信息
        List<UocSaleOrderItem> saleOrderItemInfos = this.getSaleOrderItemInfos(reqBo);
        //生成发货单id
        Long shipOrderId = IdUtil.nextId();
        //生成发货单编号
        String shipOrderNo = this.getShipOrderNo(saleOrderInfo.getOrderSource());
        //创建发货单
        this.createShipOrder(reqBo,saleOrderItemInfos,shipOrderId,shipOrderNo);
        //更新销售单发货数量
        this.updateSaleOrderSendCount(reqBo);
        //判断销售单发货是否完成
        boolean finish = this.checkSaleOrderIsFinish(reqBo);
        if(finish && !UocStateConstants.SaleOrder.XS_FH_FHWC.equals(saleOrderInfo.getSaleOrderState())){
            //如果发货完成，需更更新销售单状态
            this.updateSaleOrderState(reqBo,UocStateConstants.SaleOrder.XS_FH_FHWC);
        }else {
            //发货未完成，也要更新未发货中
            if (UocStateConstants.SaleOrder.XS_FH_DFH.equals(saleOrderInfo.getSaleOrderState())) {
                //如果发货完成，需更更新销售单状态
                this.updateSaleOrderState(reqBo,UocStateConstants.SaleOrder.XS_FH_FHZ);
            }
        }
        //返回结果
        UocSaleShipOrderRspBo rspBo = new UocSaleShipOrderRspBo();
        rspBo.setRespCode(BaseRspConstant.RSP_CODE_SUCCESS);
        rspBo.setRespDesc(BaseRspConstant.RSP_DESC_SUCCESS);
        rspBo.setShipOrderId(shipOrderId);
        rspBo.setShipOrderNo(shipOrderNo);
        rspBo.setFinish(finish);
        if (finish) {
            rspBo.setOpFlag(OP_FLAG_YES);
        }else{
            rspBo.setOpFlag(OP_FLAG_NO);
        }
        return rspBo;
    }


    /**
     * 更新销售单状态
     * @param reqBo
     * @param saleOrderState
     */
    private void updateSaleOrderState(UocSaleShipOrderReqBo reqBo, String saleOrderState) {
        UocSaleOrderDo updateSaleOrderDo = new UocSaleOrderDo();
        updateSaleOrderDo.setOrderId(reqBo.getOrderId());
        updateSaleOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
        updateSaleOrderDo.setUpdateOperId(reqBo.getUserId().toString());
        updateSaleOrderDo.setSaleOrderState(saleOrderState);
        this.iUocSaleOrderModel.updateSaleStatusAndProCode(updateSaleOrderDo);
    }

    /**
     * 判断销售单是否发货完成
     * @param reqBo
     * @return
     */
    private boolean checkSaleOrderIsFinish(UocSaleShipOrderReqBo reqBo) {
        //查询销售明细各种数量汇总
        UocSaleOrderItemQryBo uocSaleOrderItemQryBo = new UocSaleOrderItemQryBo();
        uocSaleOrderItemQryBo.setOrderId(reqBo.getOrderId());
        uocSaleOrderItemQryBo.setSaleOrderId(reqBo.getSaleOrderId());
        UocSaleOrderDo uocSaleOrderDo = this.iUocSaleOrderModel.getCollectCountImplOrderItem(uocSaleOrderItemQryBo);
        if(null != uocSaleOrderDo && !CollectionUtils.isEmpty(uocSaleOrderDo.getSaleOrderItems()) && null != uocSaleOrderDo.getSaleOrderItems().get(0)){
            //采购数量 = 发货数量 + 退货数量 则发货完成
            if (uocSaleOrderDo.getSaleOrderItems().get(0).getPurchaseCount().compareTo(uocSaleOrderDo.getSaleOrderItems().get(0).getSendCount().add(uocSaleOrderDo.getSaleOrderItems().get(0).getReturnCount())) == 0) {
                return true;
            }else{
                return false;
            }
        }else{
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARRIVA_CONFIRM_REFUSE_ERROR, "查询销售单明细数量汇总为空");
        }
    }

    /**
     * 更新销售单发货数量
     * @param reqBo
     */
    private void updateSaleOrderSendCount(UocSaleShipOrderReqBo reqBo) {
        UocSaleOrderDo updateSaleOrderDo = new UocSaleOrderDo();
        updateSaleOrderDo.setOrderId(reqBo.getOrderId());
        updateSaleOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
        updateSaleOrderDo.setUpdateOperId(String.valueOf(reqBo.getUserId()));
        updateSaleOrderDo.setUpdateTime(new Date());
        updateSaleOrderDo.setSaleOrderItems(reqBo.getShipOrderItemBoList().stream().map(uocImplShipOrderReqItemBo -> {
            UocSaleOrderItem uocSaleOrderItem = new UocSaleOrderItem();
            uocSaleOrderItem.setSaleOrderItemId(uocImplShipOrderReqItemBo.getSaleOrderItemId());
            uocSaleOrderItem.setSendCount(uocImplShipOrderReqItemBo.getSendCount());
            return uocSaleOrderItem;
        }).collect(Collectors.toList()));

        this.iUocSaleOrderModel.modifyBatchItemSendCount(updateSaleOrderDo);
    }

    /**
     * 生成发货单
     * @param reqBo
     * @param saleOrderItemInfos
     * @param shipOrderId
     * @param shipOrderNo
     */
    private void createShipOrder(UocSaleShipOrderReqBo reqBo, List<UocSaleOrderItem> saleOrderItemInfos, Long shipOrderId,String shipOrderNo) {
        //将销售单明细list转化为map
        Map<Long, UocSaleOrderItem> saleOrderItemMap = saleOrderItemInfos.stream().collect(Collectors.toMap(UocSaleOrderItem::getSaleOrderItemId, o -> o));
        //组装发货单信息
        UocShipOrderDo createShipOrderDo = JSONObject.parseObject(JSON.toJSONString(reqBo),UocShipOrderDo.class);
        //设置发货单状态
        createShipOrderDo.setShipOrderState(UocStateConstants.ShipOrder.FH_FH_FH);
        //设置发货人id
        createShipOrderDo.setShipOperId(String.valueOf(reqBo.getUserId()));
        //设置发货单创建人
        createShipOrderDo.setCreateOperId(String.valueOf(reqBo.getUserId()));
        //设置发货单创建时间
        createShipOrderDo.setCreateTime(new Date());
        //设置发货单id
        createShipOrderDo.setShipOrderId(shipOrderId);
        //设置供应商id
        createShipOrderDo.setSupNo(saleOrderItemInfos.get(0).getSupplierId());
        //设置发货单编号
        createShipOrderDo.setShipOrderNo(shipOrderNo);

        //设置发货单附件
        if (!CollectionUtils.isEmpty(reqBo.getOrderAccessoryBoList())) {
            List<UocOrderAccessory> uocOrderAccessoryList = JSONArray.parseArray(JSON.toJSONString(reqBo.getOrderAccessoryBoList())).toJavaList(UocOrderAccessory.class);
            uocOrderAccessoryList.forEach(uocOrderAccessory -> {
                uocOrderAccessory.setOrderId(reqBo.getOrderId());
                uocOrderAccessory.setCreateTime(new Date());
                uocOrderAccessory.setCreateOperId(reqBo.getUserId().toString());
                uocOrderAccessory.setId(IdUtil.nextId());
                uocOrderAccessory.setObjType(UocDicConstant.OBJ_TYPE.SHIP);
                uocOrderAccessory.setObjId(shipOrderId);
                uocOrderAccessory.setAttachmentType(UocAttachementTypeConstants.DELIVERY);
            });
            createShipOrderDo.setOrderAccessoryBoList(uocOrderAccessoryList);
        }

        //设置发货单明细
        createShipOrderDo.getShipOrderItemBoList().forEach(uocShipOrderItem -> {
            UocSaleOrderItem uocSaleOrderItem = saleOrderItemMap.get(uocShipOrderItem.getSaleOrderItemId());
            //校验是否存在已发货完的商品
            if (uocShipOrderItem.getSendCount().add(uocSaleOrderItem.getSendCount()).compareTo(uocSaleOrderItem.getPurchaseCount()) > 0) {
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_IMPL_SHIP_ORDER_ERROR, "存在已发货完成的商品，请重新选择商品发货");
            }
            //设置发货单id
            uocShipOrderItem.setShipOrderId(shipOrderId);
            //设置订单id
            uocShipOrderItem.setOrderId(reqBo.getOrderId());
            //设置销售单id
            uocShipOrderItem.setSaleOrderId(reqBo.getSaleOrderId());
            //设置销售单明细id
            uocShipOrderItem.setSaleOrderItemId(uocSaleOrderItem.getSaleOrderItemId());
            //设置计量单位
            uocShipOrderItem.setUnitName(uocSaleOrderItem.getUnitName());
            //设置创建人id
            uocShipOrderItem.setCreateOperId(String.valueOf(reqBo.getUserId()));
            //设置创建时间
            uocShipOrderItem.setCreateTime(new Date());
            //设置发货单明细id
            uocShipOrderItem.setShipOrderItemId(IdUtil.nextId());
        });
        //设置时效配置
        createShipOrderDo.setAllowChngTime(this.getShipAllowChngTime(reqBo));
        //生产发货单
        this.iUocShipOrderModel.createShipOrder(createShipOrderDo);
    }

    /**
     * 获取发货单时效配置
     * @param reqBo
     * @return
     */
    private Date getShipAllowChngTime(UocSaleShipOrderReqBo reqBo) {
        //查询销售单详情
        UocSaleOrderQryBo querySaleOrderQryBo = new UocSaleOrderQryBo();
        querySaleOrderQryBo.setSaleOrderId(reqBo.getSaleOrderId());
        querySaleOrderQryBo.setOrderId(reqBo.getOrderId());
        UocSaleOrderDo saleOrderInfo = this.iUocSaleOrderModel.qrySaleOrder(querySaleOrderQryBo);

        //查询时效类配置
        UocConfEffectiveQryBo uocConfEffectiveQryBo = new UocConfEffectiveQryBo();
        uocConfEffectiveQryBo.setEffectiveCode(UocConstant.EffectiveCode.TIME_LIMIT_FOR_ABNORMAL_ORDER_CHANGES);
        List<UocConfEffective> rows = this.iUocOrderModel.qryListPageConfEffective(uocConfEffectiveQryBo).getRows();

        Date allowChngTime = null;
        if (ObjectUtil.isNotEmpty(rows)) {
            //将时效配置转化为map
            Map<String, UocConfEffective> confEffectiveMap = rows.stream().collect(Collectors.toMap(o -> o.getBusinessTypeCode() + o.getSupplierNo(), Function.identity()));
            //找到对应配置的时效配置
            if(null != saleOrderInfo){
                String key = null;
                if(null != saleOrderInfo.getStakeholder()){
                    key = saleOrderInfo.getOrderSource() + saleOrderInfo.getStakeholder().getSupId();
                }else{
                    key = String.valueOf(saleOrderInfo.getOrderSource()) + reqBo.getSupId();
                }
                if (ObjectUtil.isNotNull(confEffectiveMap.get(key))){
                    allowChngTime = new Date(confEffectiveMap.get(key).getAging());
                }else {
                    allowChngTime = new Date(confEffectiveMap.get(saleOrderInfo.getOrderSource() + SUP_NO).getAging());
                }
            }else{
                allowChngTime = new Date(confEffectiveMap.get(EC_IMPORT + SUP_NO).getAging());
            }
        }
        return allowChngTime;
    }


    /**
     * 查询销售单明细
     */
    private List<UocSaleOrderItem> getSaleOrderItemInfos(UocSaleShipOrderReqBo reqBo) {
        List<Long> saleOrderItemIds = reqBo.getShipOrderItemBoList().stream().map(UocSaleShipOrderReqItemBo::getSaleOrderItemId).collect(Collectors.toList());
        UocSaleOrderItemQryBo uocSaleOrderItemQryBo = new UocSaleOrderItemQryBo();
        uocSaleOrderItemQryBo.setSaleOrderItemIdList(saleOrderItemIds);
        uocSaleOrderItemQryBo.setOrderId(reqBo.getOrderId());
        uocSaleOrderItemQryBo.setSaleOrderId(reqBo.getSaleOrderId());
        UocSaleOrderDo uocSaleOrderDo = this.iUocSaleOrderModel.getListSaleOrderItem(uocSaleOrderItemQryBo);
        List<UocSaleOrderItem> saleOrderItems = uocSaleOrderDo.getSaleOrderItems();
        if (CollectionUtils.isEmpty(saleOrderItems)) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_IMPL_SHIP_ORDER_ERROR, "获取销售单明细为空");
        }
        return saleOrderItems;
    }

    /**
     * 获取发货单编号
     * @param orderSource
     * @return
     */
    private String getShipOrderNo(Integer orderSource){
        UocShipOrderDo uocShipOrderDo = new UocShipOrderDo();
        uocShipOrderDo.setOrderSource(orderSource);
        uocShipOrderDo = this.iUocShipOrderModel.getShipOrderNo(uocShipOrderDo);
        if(null != uocShipOrderDo){
            return uocShipOrderDo.getShipOrderNo();
        }else{
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_IMPL_SHIP_ORDER_ERROR, "生成发货单编号失败");
        }
    }

    /**
     * 获取
     * @param reqBo
     * @return
     */
    private UocSaleOrderDo getSaleOrderInfo(UocSaleShipOrderReqBo reqBo) {
        UocSaleOrderDo uocSaleOrderDo = new UocSaleOrderDo();
        uocSaleOrderDo.setSaleOrderId(reqBo.getSaleOrderId());
        uocSaleOrderDo.setOrderId(reqBo.getOrderId());
        uocSaleOrderDo = this.iUocSaleOrderModel.getSaleOrderMain(uocSaleOrderDo);
        if (null == uocSaleOrderDo) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_IMPL_SHIP_ORDER_ERROR, "查询销售单为空");
        }
        return uocSaleOrderDo;
    }

    /**
     * 参数校验
     * @param reqBo
     */
    private void validateArg(UocSaleShipOrderReqBo reqBo) {
        if (null == reqBo) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参对象为空");
        }
        if (null == reqBo.getOrderId()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参订单ID为空");
        }
        if (null == reqBo.getSaleOrderId()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参销售单ID为空");
        }
        if (null == reqBo.getUserId()) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_ARG_EMPTY_ERROR, "入参用户ID为空");
        }

    }
}

