package com.tydic.dyc.oc.transactionservice;

import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSON;
import com.tydic.dyc.base.exception.BaseBusinessException;
import com.tydic.dyc.oc.constants.UocRspConstants;
import com.tydic.dyc.oc.constants.UocStateConstants;
import com.tydic.dyc.oc.model.insporder.IUocInspOrderModel;
import com.tydic.dyc.oc.model.insporder.UocInspOrderDo;
import com.tydic.dyc.oc.model.insporder.qrybo.UocInspOrderQryBo;
import com.tydic.dyc.oc.model.saleorder.IUocSaleOrderModel;
import com.tydic.dyc.oc.model.saleorder.UocSaleOrderDo;
import com.tydic.dyc.oc.service.domainservice.bo.UocUpdateInspPayStatusServiceReqBo;
import com.tydic.dyc.oc.service.domainservice.bo.UocUpdateInspPayStatusServiceRspBo;
import com.tydic.dyc.oc.service.domainservice.bo.UocUpdateInspPayStatusServiceRspInspBo;
import com.tydic.dyc.oc.service.domainservice.bo.UocUpdateInspPayStatusServiceRspSaleOrderBo;
import com.tydic.dyc.oc.utils.UocRu;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Propagation;
import org.springframework.transaction.annotation.Transactional;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 标题:UocUpdateInspPayStatusTransaction
 * 说明:
 * 时间:2023/9/27 11:00
 * 作者:田桂银
 */
@Component
@Slf4j
public class UocUpdateInspPayStatusTransaction {

    //业务：
    //已付过款的协议订单，确定该**订单已完成且后续不会再付款**，则由采购商城向财务系统发送订单办结的请求消息（**增加服务：验收单支付完成处理**）（**支持幂等**）
    //结算付款完成后，以验收单维度通知订单
    //订单判断该验收单所属销售单是否验收完成，同时该销售单下所属验收单都付款完成（添加字段：付款状态）
    //如果判断条件达成，则向财务系统发送订单办结的请求消息

    /**
     * 验收数量，默认100，超过100报错
     */
    @Value("${uoc.updateInspPayStatus.size:100}")
    private Integer updateInspPayStatusSize;

    @Autowired
    private IUocInspOrderModel iUocInspOrderModel;

    @Autowired
    private IUocSaleOrderModel iUocSaleOrderModel;

    @Transactional(propagation = Propagation.REQUIRES_NEW)
    public UocUpdateInspPayStatusServiceRspBo updateInspPayStatus(UocUpdateInspPayStatusServiceReqBo reqBo) {
        log.info("验收单支付回调入参：{}", JSON.toJSONString(reqBo));
        validateReqBo(reqBo);
        UocUpdateInspPayStatusServiceRspBo rspBo = UocRu.success(UocUpdateInspPayStatusServiceRspBo.class);
        List<Long> inspList = reqBo.getInspList();

        UocInspOrderQryBo qryInspBo = new UocInspOrderQryBo();
        qryInspBo.setInspOrderIdList(inspList);
        //批量查询验收单数据
        List<UocInspOrderDo> listInspOrder = iUocInspOrderModel.getListInspOrder(qryInspBo);
        List<Long> saleOrderIdList = listInspOrder.stream().map(UocInspOrderDo::getSaleOrderId).distinct().collect(Collectors.toList());

        UocSaleOrderDo qrySaleBo = new UocSaleOrderDo();
        qrySaleBo.setSaleOrderIdList(saleOrderIdList);
        //批量查询销售单数据
        List<UocSaleOrderDo> uocSaleOrderDoList = iUocSaleOrderModel.qrySaleOrderList(qrySaleBo);

        //遍历判断验收单支付状态，或更新验收单支付状态到 “已支付”
        for (UocInspOrderDo uocInspOrderDo : listInspOrder) {
            Long saleOrderId = uocInspOrderDo.getSaleOrderId();
            Long orderId = uocInspOrderDo.getOrderId();
            Integer payStatus = uocInspOrderDo.getPayStatus();
            Long inspOrderId = uocInspOrderDo.getInspOrderId();

            UocUpdateInspPayStatusServiceRspInspBo inspTmpBo = new UocUpdateInspPayStatusServiceRspInspBo();
            rspBo.getInspList().add(inspTmpBo);
            inspTmpBo.setOrderId(orderId);
            inspTmpBo.setSaleOrderId(saleOrderId);
            inspTmpBo.setInspOrderId(inspOrderId);

            if (UocStateConstants.INSP_ORDER_STATUS.PAYED.equals(payStatus)) {
                inspTmpBo.setInspPayStatus(payStatus);
            } else {
                inspTmpBo.setInspPayStatus(UocStateConstants.INSP_ORDER_STATUS.PAYED);
                //更新到已支付状态
                UocInspOrderDo updateInspBo = new UocInspOrderDo();
                updateInspBo.setOrderId(orderId);
                updateInspBo.setInspOrderId(inspOrderId);
                updateInspBo.setPayStatus(UocStateConstants.INSP_ORDER_STATUS.PAYED);
                log.info("验收单支付回调更新验收单状态入参：{}", JSON.toJSONString(updateInspBo));
                iUocInspOrderModel.updateInspPayStatus(updateInspBo);
            }
        }

        //重新查询，更新后的所有验收单数据
        List<UocInspOrderDo> updateedList = iUocInspOrderModel.getListInspOrder(qryInspBo);
        Map<Long, List<UocInspOrderDo>> saleInspMap = updateedList.stream().collect(Collectors.groupingBy(UocInspOrderDo::getSaleOrderId));
        //如果该销售单下，所有的验收单都支付完成了，且销售单状态为（已验收，拒收，部分验收）时，则需要更新销售单支付状态，同时推送nc完结消息
        for (UocSaleOrderDo uocSaleOrderDo : uocSaleOrderDoList) {
            List<UocInspOrderDo> uocInspOrderDoList = saleInspMap.get(uocSaleOrderDo.getSaleOrderId());
            //所有验收单都支付完成，默认true
            boolean allInspPayed = true;
            for (UocInspOrderDo uocInspOrderDo : uocInspOrderDoList) {
                if (!UocStateConstants.INSP_ORDER_STATUS.PAYED.equals(uocInspOrderDo.getPayStatus())) {
                    allInspPayed = false;
                    break;
                }
            }
            if (allInspPayed
                    &&
                    (UocStateConstants.SaleOrder.XS_DH_JS.equals(uocSaleOrderDo.getSaleOrderState()) ||
                    UocStateConstants.SaleOrder.XS_YS_YS.equals(uocSaleOrderDo.getSaleOrderState()) ||
                    UocStateConstants.SaleOrder.XS_YS_BFYS.equals(uocSaleOrderDo.getSaleOrderState()))
                    && !UocStateConstants.PAY_STATUS.PAYED.equals(uocSaleOrderDo.getPayState())) {

                log.info("销售单({})更新状态为已支付({})", uocSaleOrderDo.getSaleOrderId(), UocStateConstants.PAY_STATUS.PAYED);
                //更新支付状态为已支付
                UocSaleOrderDo updateSaleStateDo = new UocSaleOrderDo();
                updateSaleStateDo.setOrderId(uocSaleOrderDo.getOrderId());
                updateSaleStateDo.setSaleOrderId(uocSaleOrderDo.getSaleOrderId());
                updateSaleStateDo.setPayState(UocStateConstants.PAY_STATUS.PAYED);
                updateSaleStateDo.setUpdateTime(new Date());
                iUocSaleOrderModel.modifySaleOrderMain(updateSaleStateDo);

                UocUpdateInspPayStatusServiceRspSaleOrderBo addCompletionBO = new UocUpdateInspPayStatusServiceRspSaleOrderBo();
                addCompletionBO.setSaleOrderId(uocSaleOrderDo.getSaleOrderId());
                addCompletionBO.setOrderId(uocSaleOrderDo.getOrderId());
                rspBo.getNeedCompletionSaleOrderList().add(addCompletionBO);
            }
        }
        return rspBo;
    }

    private void validateReqBo(UocUpdateInspPayStatusServiceReqBo reqBo) {
        if (ObjectUtil.isEmpty(reqBo)) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "入参对象不能为空");
        }
        if (ObjectUtil.isEmpty(reqBo.getInspList())) {
            throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "入参对象属性[inspList]不能为空");
        } else {
            if (reqBo.getInspList().size() > updateInspPayStatusSize) {
                throw new BaseBusinessException(UocRspConstants.RSP_CODE_METHOD_ARG_EMPTY_ERROR, "入参验收单数量不能超过：" + updateInspPayStatusSize);
            }
        }

    }

}


