package com.tydic.dyc.agr.service.agr;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSON;
import com.ohaotian.plugin.common.util.MoneyUtils;
import com.ohaotian.plugin.mq.proxy.ProxyMessage;
import com.ohaotian.plugin.mq.proxy.ProxyMessageProducer;
import com.tydic.dyc.agr.model.agr.AgrAgrDo;
import com.tydic.dyc.agr.model.agr.IAgrAgrModel;
import com.tydic.dyc.agr.model.agr.qrybo.*;
import com.tydic.dyc.agr.model.agr.sub.AgrItem;
import com.tydic.dyc.agr.model.agrchange.sub.AgrItemChng;
import com.tydic.dyc.agr.service.agr.bo.AgrDealAgrItemPriceExecutionSyncReqBo;
import com.tydic.dyc.agr.service.agr.bo.AgrDealAgrItemPriceExecutionSyncRspBo;
import com.tydic.dyc.agr.service.agr.bo.AgrItemPriceExecutionBo;
import com.tydic.dyc.agr.service.agr.bo.AgrJnAgrSyncItemBo;
import com.tydic.dyc.agr.service.constants.AgrCommConstant;
import com.tydic.dyc.agr.service.constants.AgrRspConstant;
import com.tydic.dyc.agr.service.domainservice.bo.AgrSkuChangePriceBO;
import com.tydic.dyc.agr.service.domainservice.bo.AgrSyncUccReqBO;
import com.tydic.dyc.base.exception.BaseBusinessException;
import lombok.extern.ohaotian.HTServiceImpl;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.util.StringUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 标题：AgrDealAgrItemPriceExecutionSyncServiceImpl
 * 说明：京能协议明细价格执行记录信息同步接口
 * 时间：2024/7/23 14:27
 *
 * @author 周钰焱.
 */
@Slf4j
@HTServiceImpl
public class AgrDealAgrItemPriceExecutionSyncServiceImpl implements AgrDealAgrItemPriceExecutionSyncService {

    @Autowired
    private IAgrAgrModel iAgrAgrModel;

    @Value("${AGR_SKU_CHANGE_TOPIC}")
    private String agrSkuChangeTopic;

    @Value("${AGR_SKU_CHANGE_TAG}")
    private String agrSkuChangeTag;

    @Resource(name = "agrSyncSkuStatusMqServiceProvider")
    private ProxyMessageProducer agrSyncSkuStatusMqServiceProvider;

    @Override
    public AgrDealAgrItemPriceExecutionSyncRspBo dealAgrItemPriceExecutionSync(AgrDealAgrItemPriceExecutionSyncReqBo reqBo) {
        AgrDealAgrItemPriceExecutionSyncRspBo rspBo = new AgrDealAgrItemPriceExecutionSyncRspBo();
        Long agrId = getAgrId(reqBo);
        List<String> extAgrItemIds = reqBo.getAgrItem().stream().map(AgrJnAgrSyncItemBo::getExtAgrItemId).distinct().collect(Collectors.toList());
        //通过外部协议编码查询协议明细
        AgrItemListQryReqPageBo agrItemListQryReqPageBo = new AgrItemListQryReqPageBo();
        agrItemListQryReqPageBo.setAgrId(agrId);
        agrItemListQryReqPageBo.setExtAgrItemIds(extAgrItemIds);
        agrItemListQryReqPageBo.setPageNo(-1);
        agrItemListQryReqPageBo.setPageSize(-1);
        //协议明细数据
        AgrItemListQryRspPageBo agrItemList = iAgrAgrModel.getAgrItemList(agrItemListQryReqPageBo);
        Map<String, AgrItem> agrItemExtAgrItemIdMap = agrItemList.getRows().stream().collect(Collectors.toMap(AgrItem::getExtAgrItemId, item -> item));
        List<Long> agrItemIds = agrItemList.getRows().stream().map(AgrItem::getAgrItemId).distinct().collect(Collectors.toList());
        AgrItemListPriceExecutionQryReqPageBo agrItemListPriceExecutionQryReqPageBo = new AgrItemListPriceExecutionQryReqPageBo();
        agrItemListPriceExecutionQryReqPageBo.setAgrItemIds(agrItemIds);
        agrItemListPriceExecutionQryReqPageBo.setPageNo(-1);
        agrItemListPriceExecutionQryReqPageBo.setPageSize(-1);
        //已经存在的价格数据
        AgrItemListPriceExecutionQryRspPageBo agrItemPriceExecutionList = iAgrAgrModel.getAgrItemPriceExecutionList(agrItemListPriceExecutionQryReqPageBo);
        //外部价格idMap，extItemPriceExecutionId必须唯一
        Map<String, AgrItemPriceExecutionBo> existPriceExecutionMap = agrItemPriceExecutionList.getRows().stream()
                .filter(item -> item.getExtItemPriceExecutionId() != null && !item.getExtItemPriceExecutionId().isEmpty())
                .collect(Collectors.toMap(
                        AgrItemPriceExecutionBo::getExtItemPriceExecutionId,
                        item -> item,
                        (existing, replacement) -> existing // 解决重复键问题，这里选择保留已有的值
                ));
        //处理价格执行列表数据
        this.dealAgrItemListPriceExecution(reqBo, agrItemExtAgrItemIdMap, existPriceExecutionMap);
        //如果是通过，取最新一条更新明细价格
        if (2 == reqBo.getAuditStatus()) {
            //修改协议明细价格，并启用协议明细
            List<AgrItemChng> agrItemChngs = this.updateAgrItemPrice(reqBo, agrId, agrItemExtAgrItemIdMap);
            //向商品同步价格
            this.syncAgrItemAjustPriceData(agrItemChngs, agrId);
        } else if (1 == reqBo.getAuditStatus()) {
            //审核中只停用协议明细
            stopAgrItemList(agrItemList);
        } else if (3 == reqBo.getAuditStatus()) {
            //审核驳回启用协议明细
            enableAgrItemList(agrItemList);
            //恢复商品状态
            AgrSyncUccReqBO agrSyncUccReqBO = new AgrSyncUccReqBO();
            agrSyncUccReqBO.setAgrIds(Collections.singletonList(agrId));
            agrSyncUccReqBO.setAgrItemIds(agrItemIds);
            this.toUpShelf(agrSyncUccReqBO, reqBo);
        }

        rspBo.setRespCode(AgrRspConstant.RESP_CODE_SUCCESS);
        rspBo.setRespDesc(AgrRspConstant.RESP_DESC_SUCCESS);
        rspBo.setAgrId(agrId);
        rspBo.setAgrItemIds(agrItemIds);
        return rspBo;
    }

    /**
     * 处理价格执行列表数据
     *
     * @param reqBo
     * @param agrItemExtAgrItemIdMap
     * @param existPriceExecutionMap
     */
    private void dealAgrItemListPriceExecution(AgrDealAgrItemPriceExecutionSyncReqBo reqBo, Map<String, AgrItem> agrItemExtAgrItemIdMap, Map<String, AgrItemPriceExecutionBo> existPriceExecutionMap) {
        //新增协议明细价格执行记录
        List<AgrItemPriceExecutionBo> addAgrItemPriceExecutionBoList = new ArrayList<>();
        //修改协议明细价格执行记录
        List<AgrItemPriceExecutionBo> updateAgrItemPriceExecutionBoList = new ArrayList<>();
        //组装参数
        reqBo.getAgrItem().forEach(item -> {
            if (agrItemExtAgrItemIdMap.containsKey(item.getExtAgrItemId())) {
                item.getAgrItemPriceExecutionBoList().forEach(execution -> {
                    //存在则修改(只修改审核状态)
                    if (existPriceExecutionMap.containsKey(execution.getExtItemPriceExecutionId())) {
                        execution.setId(existPriceExecutionMap.get(execution.getExtItemPriceExecutionId()).getId());
                        execution.setTaxPrice(execution.getNetUnitPrice().add(agrItemExtAgrItemIdMap.get(item.getExtAgrItemId()).getTaxRate().movePointLeft(2).multiply(execution.getNetUnitPrice())));
                        execution.setAuditStatus(reqBo.getAuditStatus());
                        execution.setAuditId(reqBo.getAuditId());
                        updateAgrItemPriceExecutionBoList.add(execution);
                    } else {
                        //不存在就新增
                        execution.setAgrItemId(agrItemExtAgrItemIdMap.get(item.getExtAgrItemId()).getAgrItemId());
                        execution.setAuditStatus(reqBo.getAuditStatus());
                        execution.setAuditId(reqBo.getAuditId());
                        execution.setTaxPrice(execution.getNetUnitPrice().add(agrItemExtAgrItemIdMap.get(item.getExtAgrItemId()).getTaxRate().movePointLeft(2).multiply(execution.getNetUnitPrice())));
                        addAgrItemPriceExecutionBoList.add(execution);
                    }
                });
            }
        });


        this.iAgrAgrModel.createAgrItemPriceExecution(addAgrItemPriceExecutionBoList);
        this.iAgrAgrModel.updateAgrItemPriceExecution(updateAgrItemPriceExecutionBoList);
    }


    /**
     * 修改协议明细价格
     *
     * @param reqBo
     * @param agrId
     * @param agrItemExtAgrItemIdMap
     */
    private List<AgrItemChng> updateAgrItemPrice(AgrDealAgrItemPriceExecutionSyncReqBo reqBo, Long agrId, Map<String, AgrItem> agrItemExtAgrItemIdMap) {
        // 更新协议明细价格
        AgrAgrDo agrAgrDo = new AgrAgrDo();
        agrAgrDo.setAgrId(agrId);
        //协议明细价格修改集合
        List<AgrItem> updateAgrItemList = new ArrayList<>();
        //向商品同步价格变更集合
        List<AgrItemChng> agrItemChngs = new ArrayList<>();
        for (AgrJnAgrSyncItemBo agrJnAgrSyncItemBo : reqBo.getAgrItem()) {
            List<AgrItemPriceExecutionBo> list = agrJnAgrSyncItemBo.getAgrItemPriceExecutionBoList();
            // 使用 Stream API 找到最新日期的对象
            Optional<AgrItemPriceExecutionBo> latestItem = list.stream()
                    .max(Comparator.comparing(AgrItemPriceExecutionBo::getExecutionEffDate));
            // 如果集合不为空，获取最新日期的对象
            AgrItemPriceExecutionBo latest = latestItem.orElse(null);
            if (latest != null && agrItemExtAgrItemIdMap.containsKey(agrJnAgrSyncItemBo.getExtAgrItemId())) {
                AgrItem agrItem = new AgrItem();
                agrItem.setAgrItemId(agrItemExtAgrItemIdMap.get(agrJnAgrSyncItemBo.getExtAgrItemId()).getAgrItemId());
                BigDecimal taxPrice = latest.getNetUnitPrice().add(agrItemExtAgrItemIdMap.get(agrJnAgrSyncItemBo.getExtAgrItemId()).getTaxRate().movePointLeft(2).multiply(latest.getNetUnitPrice()));
                BigDecimal taxAmount = taxPrice.multiply(agrItemExtAgrItemIdMap.get(agrJnAgrSyncItemBo.getExtAgrItemId()).getBuyNumber());
                agrItem.setNoTaxPrice(latest.getNetUnitPrice());
                agrItem.setNoTaxAmount(latest.getNetUnitPrice().multiply(agrItemExtAgrItemIdMap.get(agrJnAgrSyncItemBo.getExtAgrItemId()).getBuyNumber()));
                agrItem.setBuyPrice(taxPrice);
                agrItem.setBuyPriceSum(taxAmount);
                agrItem.setSalePrice(taxPrice);
                agrItem.setSalePriceSum(taxAmount);
                agrItem.setTaxPrice(taxPrice);
                agrItem.setTaxAmount(taxAmount);
                agrItem.setAgrItemStatus(1);
                updateAgrItemList.add(agrItem);
                AgrItemChng agrItemChng = new AgrItemChng();
                agrItemChng.setAgrItemId(agrItemExtAgrItemIdMap.get(agrJnAgrSyncItemBo.getExtAgrItemId()).getAgrItemId());
                agrItemChng.setBuyPrice(taxPrice);
                agrItemChng.setSalePrice(taxPrice);
                agrItemChngs.add(agrItemChng);
            }
        }
        agrAgrDo.setAgrItem(updateAgrItemList);
        this.iAgrAgrModel.updateAgrItemByBatch(agrAgrDo);
        return agrItemChngs;
    }

    /**
     * 查询协议信息
     */
    private Long getAgrId(AgrDealAgrItemPriceExecutionSyncReqBo reqBO) {
        AgrAgrQryBo agrAgrQryBo = new AgrAgrQryBo();
        agrAgrQryBo.setAgrCode(reqBO.getAgrCode());
        AgrAgrDo agrAgrDo = iAgrAgrModel.getAgrMain(agrAgrQryBo);
        if (null == agrAgrDo) {
            throw new BaseBusinessException(AgrRspConstant.RESP_CODE_COMMON_PARAM_OBJ_EMPTY_ERROR, "协议信息不存在");
        }
        return agrAgrDo.getAgrId();
    }


    /**
     * 组装调价数据
     */
    private void syncAgrItemAjustPriceData(List<AgrItemChng> agrItemChngs, Long agrId) {
        AgrAgrQryBo agrAgrQryBo = new AgrAgrQryBo();
        agrAgrQryBo.setAgrId(agrId);
        AgrAgrDo agrMain = iAgrAgrModel.getAgrMain(agrAgrQryBo);
        Integer agrStatus = agrMain.getAgrStatus();
        List<AgrSkuChangePriceBO> skuChangePriceBOs = new ArrayList<>();
        agrItemChngs.forEach(agrItemChng -> {
            AgrSkuChangePriceBO agrSkuChangePriceBO = new AgrSkuChangePriceBO();
            agrSkuChangePriceBO.setAgreementId(agrId);
            agrSkuChangePriceBO.setAgreementSkuId(agrItemChng.getAgrItemId());
            agrSkuChangePriceBO.setThawFlag(AgrCommConstant.AgreementStatus.ENABLE.equals(agrStatus) ? 1 : 0);
            try {
                agrSkuChangePriceBO.setSalePrice(MoneyUtils.BigDecimal2Long(agrItemChng.getSalePrice()));
                agrSkuChangePriceBO.setBuyPrice(MoneyUtils.BigDecimal2Long(agrItemChng.getBuyPrice()));
            } catch (Exception e) {
                log.info(e.getMessage());
            }
            skuChangePriceBOs.add(agrSkuChangePriceBO);
        });
        Map<String, Object> content = new HashMap<>();
        content.put("skuChangePriceBOs", skuChangePriceBOs);
        content.put("changeType", AgrCommConstant.AgrChangeOperType.PRICE);
        // 发消息
        this.sendMsg(JSON.toJSONString(content));

    }

    /**
     * 停用协议明细
     *
     * @param agrItemList
     */
    private void stopAgrItemList(AgrItemListQryRspPageBo agrItemList) {
        AgrAgrDo agrAgrDo = new AgrAgrDo();
        List<AgrItem> updateAgrItemList = new ArrayList<>();
        for (AgrItem row : agrItemList.getRows()) {
            AgrItem agrItem = new AgrItem();
            agrItem.setAgrItemId(row.getAgrItemId());
            agrItem.setAgrItemStatus(0);
            updateAgrItemList.add(agrItem);
        }
        agrAgrDo.setAgrItem(updateAgrItemList);
        this.iAgrAgrModel.updateAgrItemByBatch(agrAgrDo);
    }

    /**
     * 启用协议明细
     *
     * @param agrItemList
     */
    private void enableAgrItemList(AgrItemListQryRspPageBo agrItemList) {
        AgrAgrDo agrAgrDo = new AgrAgrDo();
        List<AgrItem> updateAgrItemList = new ArrayList<>();
        for (AgrItem row : agrItemList.getRows()) {
            AgrItem agrItem = new AgrItem();
            agrItem.setAgrItemId(row.getAgrItemId());
            agrItem.setAgrItemStatus(1);
            updateAgrItemList.add(agrItem);
        }
        agrAgrDo.setAgrItem(updateAgrItemList);
        this.iAgrAgrModel.updateAgrItemByBatch(agrAgrDo);
    }

    /**
     * 恢复商品状态
     */
    private void toUpShelf(AgrSyncUccReqBO agrSyncUccReqBO, AgrDealAgrItemPriceExecutionSyncReqBo reqBO) {
        Map<String, Object> content = new HashMap<>();
        List<Long> agreementIds = new ArrayList<>();
        List<Long> agreementItemIds = new ArrayList<>();
        //调用协议详情查询方法
        AgrAgrQryBo agrAgrQryBo = new AgrAgrQryBo();
        agrAgrQryBo.setAgrId(getAgrId(reqBO));
        AgrAgrDo agrDetail = this.iAgrAgrModel.getAgrDetail(agrAgrQryBo);
        //框架协议校验协议状态、明细状态、变价数据
        if (agrDetail.getAgrType().equals(AgrCommConstant.AgrType.AGR)) {
            List<AgrItem> agrItems = iAgrAgrModel.checkAgrItemUnfreeze(agrSyncUccReqBO);
            if (CollectionUtil.isNotEmpty(agrItems)) {
                agreementIds = agrItems.stream().map(AgrItem::getAgrId).distinct().collect(Collectors.toList());
                agreementItemIds = agrItems.stream().map(AgrItem::getAgrItemId).distinct().collect(Collectors.toList());
            } else {
                return;
            }
        } else {
            if (CollectionUtil.isNotEmpty(agrSyncUccReqBO.getAgrItemIds())) {
                agreementItemIds.addAll(agrSyncUccReqBO.getAgrItemIds());
            }
            agreementIds.add(agrSyncUccReqBO.getAgrId());
        }
        content.put("changeType", AgrCommConstant.AgrChangeOperType.THAW);
        content.put("agreementIds", agreementIds.stream().distinct().collect(Collectors.toList()));
        content.put("agreementSkuIds", agreementItemIds.stream().distinct().collect(Collectors.toList()));
        content.put("operId", agrSyncUccReqBO.getOperId());
        content.put("operName", agrSyncUccReqBO.getOperName());
        // 发消息
        this.sendMsg(JSON.toJSONString(content));
    }

    /**
     * 发送消息
     */
    private void sendMsg(String content) {
        if (StringUtils.hasText(content)) {
            log.info("发消息同步商品商品中心入参为" + content);
            agrSyncSkuStatusMqServiceProvider.send(new ProxyMessage(agrSkuChangeTopic, agrSkuChangeTag, content));
            log.info("发消息同步商品商品中心成功");
        }

    }
}
