package com.tydic.dyc.pro.base.core.apollo;

import org.springframework.boot.context.event.ApplicationPreparedEvent;
import org.springframework.context.ApplicationListener;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.MutablePropertySources;
import org.springframework.core.env.PropertySource;

/**
 * 在 ApplicationPreparedEvent 时机，对后加入的属性源（如 Apollo）进行二次 wrap，
 * 让其中的 ENC(...) 在绑定/读取时也能解密。
 */
public class LateDecryptingPropertySourcesListener implements ApplicationListener<ApplicationPreparedEvent> {

    private static final String BOOT_BINDER_PS_PREFIX = "org.springframework.boot.context.properties.source";
    private static final String TOP_NAME = "encDecTop";

    @Override
    public void onApplicationEvent(ApplicationPreparedEvent event) {
        ConfigurableEnvironment env = event.getApplicationContext().getEnvironment();
        MutablePropertySources sources = env.getPropertySources();
        int wrapped = 0, skipped = 0;

        System.out.println("[ENC-DEC-LATE] scanning property sources for wrapping ...");
        for (PropertySource<?> ps : sources) {
            String cn = ps.getClass().getName();
            // 跳过 Spring Boot Binder 的聚合源，避免循环委托
            if (cn != null && cn.startsWith(BOOT_BINDER_PS_PREFIX)) {
                System.out.println("[ENC-DEC-LATE] skip binder: " + ps.getName() + " {" + cn + "}");
                skipped++;
                continue;
            }
            // 跳过我们自己插入的顶层代理
            if (TOP_NAME.equals(ps.getName())) {
                skipped++;
                continue;
            }
            // 跳过已经包裹过的解密源
            if (ps instanceof DecryptingPropertySource || ps instanceof DecryptingPropertySource.EnumerableWrapper) {
                skipped++;
                continue;
            }

            PropertySource<?> wrappedPs = DecryptingPropertySource.wrap(ps);
            if (wrappedPs != ps) {
                sources.replace(ps.getName(), wrappedPs);
                System.out.println("[ENC-DEC-LATE] wrap: " + ps.getName() + " -> " + wrappedPs.getClass().getSimpleName());
                wrapped++;
            } else {
                System.out.println("[ENC-DEC-LATE] passthrough: " + ps.getName() + " {" + ps.getClass().getSimpleName() + "}");
                skipped++;
            }
        }
        System.out.println("[ENC-DEC-LATE] done. wrapped=" + wrapped + ", skipped=" + skipped);
    }
}